# -*- coding: utf-8 -*-

from functools import (
    partial,
    wraps,
)

import flask
from passport.backend.social.api.common import database_error
from passport.backend.social.api.views.v2.grants import (
    access_denied,
    get_grants_context,
)
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import (
    DatabaseError,
    GrantsMissingError,
)
from passport.backend.social.proxylib.error_handler import ErrorHandler as _ErrorHandler


def handler_v2_from_view():
    def decorate(func):
        return HandlerV2WithCustomViewFunction.adapt_view(func)
    return decorate


class HandlerV2(object):
    def __init__(self, request):
        self._request = request

    @classmethod
    def as_view(cls):
        view_func = partial(cls._view_func, cls)
        view_func.__name__ = cls.__name__
        return view_func

    @staticmethod
    def _view_func(cls):
        handler = cls(flask.request)
        return handler.process_request()

    def process_request(self):
        try:
            request_ctx.grants_context = get_grants_context()
            response = self._process_request()
        except Exception as e:
            response = self._process_exception(e)
        return response

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def _process_exception(self, exception):
        error_handler = ErrorHandler(exception)
        response = error_handler.exception_to_response()
        if not response:
            raise
        error_handler.exception_to_graphite()
        return response


class HandlerV2WithCustomViewFunction(HandlerV2):
    def __init__(self, request, view_func):
        super(HandlerV2WithCustomViewFunction, self).__init__(request)
        self._view_func = view_func

    @classmethod
    def adapt_view(cls, view_func):
        wrapper = partial(cls._view_func, cls, view_func)
        wrapper = wraps(view_func)(wrapper)
        return wrapper

    @staticmethod
    def _view_func(cls, view_func, **kwargs):
        view_func = partial(view_func, **kwargs)
        handler = cls(flask.request, view_func)
        return handler.process_request()

    def _process_request(self):
        return self._view_func()


class ErrorHandler(_ErrorHandler):
    def exception_to_response(self):
        if isinstance(self._exception, DatabaseError):
            return database_error()
        if isinstance(self._exception, GrantsMissingError):
            return access_denied(self._exception)
