# -*- coding: utf-8 -*-


from flask import request
from passport.backend.social.api.common import (
    error,
    filter_allowed_grants,
    internal_error,
    jsonify_code,
)
from passport.backend.social.api.views.v2.base import handler_v2_from_view
from passport.backend.social.api.views.v2.grants import build_grants_checking_decorator as grants
from passport.backend.social.common.builders.blackbox import Blackbox
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.profile import (
    BaseProfileCreationError,
    create_profile,
    DatabaseFailedProfileCreatetionError,
)
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.task import (
    delete_task_from_redis,
    load_task_from_redis,
)
from passport.backend.social.common.token.utils import (
    get_grants_to_read_token,
    get_grants_to_use_token,
)
from passport.backend.social.common.useragent import get_http_pool_manager


def _get_task(task_id):
    return load_task_from_redis(get_redis(),  task_id)


@handler_v2_from_view()
@grants('task-get')
def get_task(task_id):
    task = _get_task(task_id)
    if not task:
        return error(code=404, name='task-not-found', description='Task with the given task_id not found')
    grants.check_any_of_grants(get_grants_to_use_token(application=task.application))
    return jsonify_code(task.get_dict_for_response(
        with_token=bool(filter_allowed_grants(get_grants_to_read_token(application=task.application))),
    ))


@handler_v2_from_view()
@grants('task-delete')
def delete_task(task_id):
    result = delete_task_from_redis(get_redis(), task_id)
    return jsonify_code(dict(status='ok', deleted=result))


@handler_v2_from_view()
@grants('task-get')
@grants('profile-create')
def create_profile_task(task_id):
    task = _get_task(task_id)
    if not task:
        return error(code=404, name='task-not-found', description='Task with the given task_id not found')

    uid = request.values.get('uid')
    if uid:
        grants.check_any_of_grants(['task-uid'])

    uid = uid or task.uid
    if not uid:
        return error(name='uid.empty', description='"uid" should be passed through either parameters or task')

    subscription_id = request.values.get('subscription_id') or task.sid
    if subscription_id is not None:
        try:
            subscription_id = int(subscription_id)
            if subscription_id < 0:
                raise ValueError
        except ValueError:
            return error(name='subsciption_id-invalid', description='subscription_id should be unsigned integer')

    allow_auth = request.values.get('allow_auth', 0)
    if allow_auth is not None:
        grants.check_any_of_grants(['task-allow_auth'])
        try:
            allow_auth = int(allow_auth)
            if allow_auth not in [0, 1]:
                raise ValueError
        except ValueError:
            return error(name='allow_auth-invalid', description='allow_auth should be 0 or 1')

    try:
        new_profile_id = create_profile(
            get_slave_engine(),
            get_master_engine(),
            uid,
            task.get_social_userinfo(),
            task.get_token(),
            task.finished,
            None,
            subscription_id,
            allow_auth,
            refresh_token=task.get_refresh_token(),
            blackbox=Blackbox(http_pool_manager=get_http_pool_manager()),
        )
    except DatabaseFailedProfileCreatetionError:
        return internal_error('Database failed')
    except BaseProfileCreationError as e:
        return error(name='task-invalid', description=e.description)

    return jsonify_code(dict(uid=uid, status='ok', profile_id=new_profile_id))
