# -*- coding: utf-8 -*-

import logging
from operator import itemgetter

from flask import request
from passport.backend.core import Undefined
from passport.backend.core.types.ip.ip import IP
from passport.backend.social.api.common import (
    application_unknown_error,
    error,
    execute,
    executew,
    get_current_subscription_state,
    get_profiles_from_db_batch,
    get_timestamp,
    internal_error,
    invalidate_billing_cache,
    jsonify_code,
    not_found,
    provider_unknown_error,
    required_args,
)
from passport.backend.social.api.statbox import StatboxLogger
from passport.backend.social.api.views.v2.base import (
    HandlerV2,
    handler_v2_from_view,
)
from passport.backend.social.api.views.v2.grants import build_grants_checking_decorator as grants
from passport.backend.social.common.builders.blackbox import consider_authenticated
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.schemas import (
    person_table as pert,
    profile_table as pt,
    sub_table as st,
)
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.exception import (
    FailureSourceType,
    GrantsMissingError,
    NetworkProxylibError,
    NotFound,
    ProviderTemporaryUnavailableProxylibError,
    UnexpectedResponseProxylibError,
    UnrefreshableTokenError,
)
from passport.backend.social.common.limits import get_qlimits
from passport.backend.social.common.misc import (
    get_business_userid,
    GraphiteMessageType,
    name_for_graph_log,
    split_scope_string,
    write_graph_log_message,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.providers.Facebook import Facebook
from passport.backend.social.common.providers.Kinopoisk import Kinopoisk
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.serialize import serialize_datetime
from passport.backend.social.common.social_logging import (
    BindingLogger,
    BindingsDeletedStatboxEvent,
)
from passport.backend.social.common.token.domain import Token
from passport.backend.social.common.token.utils import (
    delete_all_tokens_for_account,
    delete_token_by_token_id,
    find_all_tokens_for_account,
    find_all_tokens_for_profile,
    find_token_by_token_id,
    get_grants_to_read_token,
    get_grants_to_update_token,
    save_token,
)
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.web_service import (
    AccountGetter,
    Response,
    YandexTokenInvalidWebServiceError,
)
from passport.backend.social.proxylib import get_proxy
from passport.backend.social.proxylib.refresh_token import (
    filter_and_refresh_newest_token,
    refresh_token_by_token,
)
from sqlalchemy.sql.expression import (
    delete,
    insert,
    or_,
    select,
    update,
)


logger = logging.getLogger('social.api.views')
exc_logger = logging.getLogger('exception')


# ============= PROFILE ==============
@handler_v2_from_view()
@grants('profile-read')
def get_profile(profile_id):
    profiles = get_profiles_from_db_batch(clause=(pt.c.profile_id == profile_id))
    if not profiles:
        return not_found(description='Profile was not found for id=`%s`' % profile_id)
    return jsonify_code(dict(profile=profiles[0]))


@handler_v2_from_view()
@required_args(profile_id=int)
def edit_profile(profile_id=''):
    profile = execute(select([pt]).where(pt.c.profile_id == profile_id)).fetchone()
    if profile is None:
        return not_found()

    grants.check_any_of_grants(['profile-update', 'profile-update-%s' % profile['provider_id']])

    values = {}
    try:
        verified = get_timestamp()
        confirmed = get_timestamp('confirmed', default_current_timestamp=True)
    except ValueError as e:
        return error(name=e.message)

    username = request.form.get('username', None)

    if verified:
        values['verified'] = serialize_datetime(verified)

    if confirmed:
        values['confirmed'] = serialize_datetime(confirmed)

    if username:
        values['username'] = username

    allow_auth = _get_allow_auth()
    if allow_auth is not None:
        values['allow_auth'] = allow_auth

    logger.debug('Updating profile fields: ' + str(values))
    if values:
        executew(update(pt).where(pt.c.profile_id == profile_id).values(**values))
    profiles = get_profiles_from_db_batch(clause=(pt.c.profile_id == profile_id))
    if not profiles:
        return not_found()
    return jsonify_code(dict(profile=profiles[0]))


def _get_allow_auth():
    allow_auth = request.values.get('allow_auth')
    if allow_auth is None:
        return
    try:
        allow_auth = int(allow_auth)
    except ValueError:
        return
    if allow_auth not in [0, 1]:
        return
    try:
        grants.check_any_of_grants(['profile-update-allow_auth'])
    except GrantsMissingError:
        return
    return allow_auth


@handler_v2_from_view()
@required_args(profile_id=int)
def delete_profile(profile_id=''):
    profile = execute(select([pt]).where(pt.c.profile_id == profile_id)).fetchone()
    if profile is None:
        return not_found()

    grants.check_any_of_grants(['profile-delete', 'profile-delete-%s' % profile['provider_id']])

    executew(delete(pt).where(pt.c.profile_id == profile_id))

    if profile.provider_id == Kinopoisk.id:
        statbox = StatboxLogger()
        statbox.log(action='update_account_yandex_bindings', uid=profile.uid)
        invalidate_billing_cache(profile.uid)
        statbox.log(action='update_account_yandex_bindings', uid=profile.userid)
        invalidate_billing_cache(profile.userid)
    elif profile.provider_id == Yandex.id:
        StatboxLogger().log(action='update_account_yandex_bindings', uid=profile.uid)
        invalidate_billing_cache(profile.uid)
        BindingLogger().log_event(
            BindingsDeletedStatboxEvent(
                master_uid=profile.uid,
                slave_provider_userids=[(Yandex.code, profile.userid)],
            ),
        )
    return ''

# ============= END PROFILE ==============

# ============= PROFILES ==============


def _merge_clauses(old_clause, new_clause):
    if old_clause is None:
        return new_clause
    if new_clause is None:
        return old_clause

    return old_clause & new_clause


@handler_v2_from_view()
@grants('profile-list')
def get_profiles(uid=None):
    uid = uid or request.args.get('uid')
    uids = request.args.get('uids')
    if uids:
        uids = [x.strip() for x in uids.split(',') if x.strip()]
    else:
        uids = []

    provider_id = request.args.get('provider_id')
    provider = request.args.get('provider')

    if provider:
        provider_info = providers.get_provider_info_by_name(provider)
        if provider_id is not None and provider_id != provider_info['id']:
            description = '`provider_id` conflicts with `provider`'
            return error(code=403, name='invalid-attributes', description=description)
        provider_id = provider_info['id']

    userid = request.args.get('userid')

    if (provider_id is None) != (userid is None):
        description = 'Both "userid" and ("provider_id" or "provider") or none of them should be set'
        return error(code=403, name='missing-attributes', description=description)

    if uid is None and not uids and not userid:
        description = 'Missed required attribute `uid` or `uids` or both `userid` and `provider_id`'
        return error(code=403, name='missing-attributes', description=description)

    if uid is not None and not uids and userid is None:
        profiles = get_profiles_from_db_batch(clause=(pt.c.uid == uid))
        return jsonify_code(dict(profiles=profiles))

    if uid is not None and uids:
        uids.append(uid)

    clause = None

    if userid:
        provider_id = int(provider_id)
        business_token = request.args.get('business_token')
        business_id = request.args.get('business_id')

        if provider_id == Facebook.id and business_id and business_token:
            bt_userid = get_business_userid(business_id, business_token)
            clause = (
                or_(pt.c.userid == userid, pt.c.userid == bt_userid) &
                (pt.c.provider_id == provider_id)
            )
        else:
            clause = (pt.c.userid == userid) & (pt.c.provider_id == provider_id)

    if uids:
        clause = _merge_clauses(clause, pt.c.uid.in_(uids))
    elif uid is not None:
        clause = _merge_clauses(clause, pt.c.uid == uid)

    profiles = get_profiles_from_db_batch(clause=clause)
    return jsonify_code(dict(profiles=profiles))


@handler_v2_from_view()
@grants('profile-search')
def search_profiles():
    email = request.args.get('email')
    username = request.args.get('username')

    if not email and not username:
        description = 'Missed required attribute `email` or `username` or both of them.'
        return error(code=403, name='missing-attributes', description=description)

    clause = None
    if email:
        clause = (pert.c.email == email)

    if username:
        clause = _merge_clauses(clause, pt.c.username == username)

    query = (
        select(
            [pt],
            from_obj=[pt.join(pert)],
        )
        .where(clause)
        .limit(get_qlimits()['profiles'])
    )
    profiles = get_profiles_from_db_batch(query=query)
    return jsonify_code(dict(profiles=profiles))


@handler_v2_from_view()
@grants('user-delete')
def delete_user(uid):
    logger.info('Deleting profiles for uid=%s' % uid)
    profiles = execute(select([pt]).where(pt.c.uid == uid)).fetchall()
    if not profiles:
        logger.info('Count of profiles deleted: 0')
        return ''

    rowcount = executew(delete(pt).where(pt.c.uid == uid)).rowcount

    statbox = StatboxLogger()
    kinopoisk_profiles = [p for p in profiles if p.provider_id == Kinopoisk.id]
    yandex_profiles = [p for p in profiles if p.provider_id == Yandex.id]
    if kinopoisk_profiles or yandex_profiles:
        statbox.log(action='update_account_yandex_bindings', uid=uid)
        invalidate_billing_cache(uid)
    for profile in kinopoisk_profiles:
        statbox.log(action='update_account_yandex_bindings', uid=profile.userid)
        invalidate_billing_cache(profile.userid)

    if yandex_profiles:
        slave_provider_userids = [(Yandex.code, p.userid) for p in yandex_profiles]
        BindingLogger().log_event(
            BindingsDeletedStatboxEvent(
                master_uid=uid,
                slave_provider_userids=slave_provider_userids,
            ),
        )

    logger.info('Count of profiles deleted: %d' % rowcount)
    return ''

# ============= END PROFILES ==============


# ============= TOKEN ==============
@handler_v2_from_view()
@grants('token-read')
@required_args(token_id=int)
def get_token(token_id):
    token = find_token_by_token_id(token_id, get_slave_engine())

    if token:
        grants.check_any_of_grants(get_grants_to_read_token(token))

    if token is not None and token.is_going_to_expire():
        try:
            refresh_token_by_token(token)
        except UnrefreshableTokenError:
            token = None
        except NetworkProxylibError:
            # Учитывая тот факт, что ручка и раньше могла отдать
            # недействительный токен, а также во имя отказоустойчивости
            # social-api к отказам внешних систем игнорируем сетевые отказы.
            pass

    if token is None:
        return not_found(name='token-not-found', description='Token was not found')
    else:
        return jsonify_code(dict(token=token.to_json_dict()))


def _get_oauth_token_and_ip_from_request(request_):
    yandex_authorization = request_.headers.get('Ya-Consumer-Authorization') or ''
    yandex_authorization = yandex_authorization.strip()
    if yandex_authorization == '':
        return None, None

    token_prefix = 'Bearer '
    if not yandex_authorization.lower().startswith(token_prefix.lower()):
        return Undefined, None

    yandex_authorization = yandex_authorization[len(token_prefix):]

    user_ip = request_.headers.get('Ya-Consumer-Client-Ip')
    user_ip = user_ip and IP(user_ip)

    consumer_ip = request_.headers.get('X-Real-Ip') or request_.remote_addr
    consumer_ip = consumer_ip and IP(consumer_ip)

    return yandex_authorization, user_ip or consumer_ip


@handler_v2_from_view()
@grants('token-read')
def get_token_newest():
    profile_id = request.args.get('profile_id')
    uid = request.args.get('uid')

    application_id = request.args.get('application_id')
    if application_id is None:
        application_name = request.args.get('application_name')
        if application_name is None:
            return error(name='application_id-empty')
        application = providers.get_application_by_name(application_name)
    else:
        application = providers.get_application_by_id(application_id)

    if not application:
        return application_unknown_error()
    application_id = application.identifier
    request_ctx.application = application

    oauth_token, user_ip = _get_oauth_token_and_ip_from_request(request)

    if oauth_token is not None and uid is None and profile_id is None:
        with consider_authenticated():
            required_grants = get_grants_to_read_token(application=application)
    else:
        required_grants = get_grants_to_read_token(application=application)
    grants.check_any_of_grants(required_grants)

    if oauth_token is not None and oauth_token is not Undefined:
        account_getter = AccountGetter(get_http_pool_manager(), request)
        try:
            account = account_getter.get_account_from_token(oauth_token, user_ip)
            uid = account.uid
            logger.debug('Uid = %s' % uid)
        except YandexTokenInvalidWebServiceError:
            return error('invalid-oauth-token', 'Invalid OAuth token')
        except:
            return internal_error('Blackbox failed')

    if profile_id is None and uid is None:
        return error(
            name='profile_id-empty',
            description='GET argument `profile_id` or `uid` or Ya-Consumer-Authorization headers with bearer-scheme is required',
        )

    if uid is not None:
        tokens = find_all_tokens_for_account(uid, get_slave_engine(), [application_id])
    else:
        tokens = find_all_tokens_for_profile(profile_id, get_slave_engine(), [application_id])

    required_scopes = split_scope_string(request.args.get('scope'))
    tokens = [t for t in tokens if t.has_every_scope(required_scopes)]

    try:
        newest_token = filter_and_refresh_newest_token(tokens)
    except NotFound:
        return not_found(name='token-not-found', description='Token was not found')
    except ProviderTemporaryUnavailableProxylibError:
        write_graph_log_message(GraphiteMessageType.error, request_ctx.handler_id, name_for_graph_log(application), FailureSourceType.external)
        return internal_error('Provider failed', code=502)
    except NetworkProxylibError:
        write_graph_log_message(GraphiteMessageType.error, request_ctx.handler_id, name_for_graph_log(application), FailureSourceType.network)
        return internal_error('Network failed', code=504)
    except UnexpectedResponseProxylibError:
        write_graph_log_message(GraphiteMessageType.error, request_ctx.handler_id, name_for_graph_log(application), FailureSourceType.external)
        return internal_error('Unexpected provider response', code=502)

    return jsonify_code(dict(token=newest_token.to_json_dict()))


@handler_v2_from_view()
@grants('token-create')
@required_args(profile_id=int, application=str, value=str, secret=str)
def create_token(profile_id='', application='', value='', secret=''):
    # secret нужен для токенов OAuth1, но игнорируется для токенов OAuth2
    app_name = application

    if not value:
        return error(name='value-empty', description='Parameter `value` is required')

    profile = execute(select([pt], pt.c.profile_id == profile_id)).fetchone()
    if profile is None:
        return not_found()

    app = providers.get_application_by_name(app_name)

    if not app and app_name.isdigit():
        app = providers.get_application_by_id(app_name)

    if not app:
        return application_unknown_error()

    grants.check_any_of_grants(get_grants_to_update_token(application=app))

    for token in find_all_tokens_for_profile(profile_id, get_slave_engine(), [app.identifier]):
        if token.value == value:
            break
    else:
        token = Token()

    try:
        verified = get_timestamp()
        confirmed = get_timestamp('confirmed')
    except ValueError as e:
        return error(name=e.message)
    if verified:
        token.verified = verified
    if confirmed:
        token.confirmed = confirmed

    if token.token_id:
        if verified or confirmed:
            save_token(token, get_master_engine())

        return jsonify_code(dict(token=token.to_json_dict()))

    token.application = app
    token.application_id = app.identifier
    token.profile_id = profile_id
    token.scopes = request.form.get('scope')
    token.secret = secret
    token.uid = profile.uid
    token.value = value
    save_token(token, get_master_engine())

    return jsonify_code(dict(token=token.to_json_dict()), code=201)


@handler_v2_from_view()
@grants('token-update')
@required_args(token_id=int)
def edit_token(token_id=''):
    token = find_token_by_token_id(token_id, get_slave_engine())
    if token is None:
        return not_found(name='token-not-found')

    grants.check_any_of_grants(get_grants_to_update_token(token))

    scopes = split_scope_string(request.form.get('scope'))
    secret = request.form.get('secret')
    verified = request.form.get('verified')
    confirmed = request.form.get('confirmed')

    if not filter(None, [scopes, secret, verified, confirmed]):
        return error(name='values-empty', description='No one possible arguments are passed')

    try:
        if verified:
            token.verified = get_timestamp()
        if confirmed:
            token.confirmed = get_timestamp('confirmed')
    except ValueError as e:
        return error(name=e.message)

    if scopes:
        token.scopes = scopes
    if secret:
        token.secret = secret

    save_token(token, get_master_engine())
    return jsonify_code(dict(token=token.to_json_dict()))


@handler_v2_from_view()
@grants('token-delete')
@required_args(token_id=int)
def delete_token(token_id):
    token = find_token_by_token_id(token_id, get_slave_engine())
    if token is None:
        return not_found(name='token-not-found')

    grants.check_any_of_grants(get_grants_to_update_token(token))

    deleted_rows = delete_token_by_token_id(token.token_id, get_master_engine())
    if not deleted_rows:
        return error(name='token-not-deleted')

    return ''


class DeleteTokensFromAccount(HandlerV2):
    def __init__(self, *args, **kwargs):
        super(DeleteTokensFromAccount, self).__init__(*args, **kwargs)
        self.apps = None
        self.uid = None

    def _process_request(self):
        grants.check_any_of_grants(['token-delete'])

        self.uid = self._request.form.get('uid')
        if self.uid is None:
            return error(name='uid-empty')

        application_name = self._request.form.get('application_name')
        provider_name = self._request.form.get('provider_name')

        if (
            application_name is None and
            provider_name is None
        ):
            return error(name='application_name-empty')

        self.apps = list()

        if application_name is not None:
            app = providers.get_application_by_name(application_name)
            if not app:
                return application_unknown_error()
            self.apps += [app]

        if provider_name is not None:
            provider_info = providers.get_provider_info_by_name(provider_name)
            if not provider_info:
                return provider_unknown_error()
            self.apps += providers.get_many_applications_by_provider(provider_name)

        for app in self.apps:
            grants.check_any_of_grants(get_grants_to_update_token(application=app))

        if self._request.form.get('revoke') == '1':
            self.revoke_tokens()

        delete_all_tokens_for_account(
            self.uid,
            application_ids=[a.identifier for a in self.apps],
            read_conn=get_slave_engine(),
            write_conn=get_master_engine(),
        )
        return ''

    def revoke_tokens(self):
        # TODO Отозвать рефреш-токены, когда понадобиться отзывать токены
        # провайдеров умеющих рефреш-токены

        for token in find_all_tokens_for_account(
            uid=self.uid,
            db=get_slave_engine(),
            application_ids=[a.identifier for a in self.apps],
        ):
            proxy = get_proxy(access_token=token.to_dict_for_proxy(), app=token.application)
            if hasattr(proxy, 'revoke_token'):
                proxy.revoke_token()


@handler_v2_from_view()
@grants('token-read')
def is_token_available():
    uid = request.values.get('uid')
    if uid is None:
        return error(name='uid-empty')

    application_names = request.values.get('application_names')
    if application_names is None:
        return error(name='application_names-empty')
    application_names = application_names.split()
    applications, _ = providers.get_many_applications_by_names(application_names)

    application_ids = set([a.identifier for a in applications])
    all_tokens = find_all_tokens_for_account(uid, get_slave_engine(), application_ids)
    alive_tokens = [t for t in all_tokens if not t.is_going_to_expire()]

    app_id_to_alive_tokens = dict()
    for token in alive_tokens:
        app_tokens = app_id_to_alive_tokens.setdefault(token.application_id, [])
        app_tokens.append(token)

    retval = []
    for app in applications:
        app_tokens = app_id_to_alive_tokens.get(app.identifier, [])
        if app_tokens:
            retval.append(dict(application_name=app.name))

    retval.sort(key=itemgetter('application_name'))
    return jsonify_code(retval, code=200)


# ============= END TOKEN ==============

# ============= SUBSCRIPTION ==============


def sub_grants(prefix):
    def wrapper(profile_id, sid):
        return ['%s-%s' % (prefix, sid)]

    return wrapper


@handler_v2_from_view()
@grants('subscription-create', sub_grants('subscription-create'))
def create_subscription(profile_id, sid):
    if execute(select([pt]).where(pt.c.profile_id == profile_id)).fetchone() is None:
        return not_found()

    sid_query = (st.c.profile_id == profile_id) & (st.c.sid == sid)
    db_subscription, is_subscribed_by_default, is_currently_subscribed = get_current_subscription_state(sid_query, sid)

    if is_currently_subscribed:
        return error(code=409, name='subscription-exists')

    values_dict = dict(profile_id=profile_id, sid=sid, value=1)

    if db_subscription and is_subscribed_by_default:
        # Здесь обязательно value=0, надо просто удалить запись,
        # тогда по дефолту подписка будет включена.
        executew(delete(st).where(sid_query))
    elif db_subscription and not is_subscribed_by_default:
        # Запись уже есть, но value совпадает с дефолтным.
        executew(update(st).values(value=1).where(sid_query))
    else:
        # Значение по умолчанию 0, добавим запись с value=1
        executew(insert(st).values(**values_dict))

    return jsonify_code(dict(subscription=values_dict), code=201)


@handler_v2_from_view()
@grants('subscription-delete', sub_grants('subscription-delete'))
def delete_subscription(profile_id, sid):
    if execute(select([pt]).where(pt.c.profile_id == profile_id)).fetchone() is None:
        return not_found()

    sid_query = (st.c.profile_id == profile_id) & (st.c.sid == sid)
    db_subscription, is_subscribed_by_default, is_currently_subscribed = get_current_subscription_state(sid_query, sid)

    if not is_currently_subscribed:
        return not_found(name='subscription-not-found')

    if db_subscription and is_subscribed_by_default:
        # В базе value=1, что совпадает с дефолтным значением.
        updated_count = executew(update(st).values(value=0).where(sid_query))
    elif db_subscription and not is_subscribed_by_default:
        # В базе value=1, но по умолчанию подписка выключена. Удалим запись.
        updated_count = executew(delete(st).where(sid_query))
    else:
        # По умолчанию подписка включена.
        updated_count = executew(insert(st).values(profile_id=profile_id, sid=sid, value=0))

    if updated_count:
        return ''
    return error(name='subscription-not-deleted')

# ============= END SUBSCRIPTION ==============


@handler_v2_from_view()
def ping_view():
    return Response('Pong\n', status=200)
