# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core import Undefined
from passport.backend.core.logging_utils.helpers import MASK_PLACEHOLDER
from passport.backend.social.api.views.v3.base import InternalApiHandlerV3
from passport.backend.social.common import validators
from passport.backend.social.common.application import (
    Application,
    ApplicationGroupId,
    build_application_name,
)
from passport.backend.social.common.builders.kolmogor import (
    KolmogorPermanentError,
    KolmogorTemporaryError,
    SOCIALISM_KOLMOGOR_NAMESPACE,
)
from passport.backend.social.common.counter import CreateApplicationByIpCounterDescriptor
from passport.backend.social.common.misc import remove_undefined_values
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.web_service import (
    ApplicationUnknownWebServiceError,
    KolmogorFailedWebServiceError,
    RateLimitExceededWebServiceError,
)
from passport.backend.utils.common import remove_none_values


logger = logging.getLogger(__name__)

_STATION_DEFAULT_DOMAIN = 'social.yandex.net'

_FORM_NAME_TO_MODEL_NAME = {
    'application_display_name': 'display_name',
    'application_name': 'name',
    'authorization_url': 'authorization_url',
    'client_id': 'id',
    'client_secret': 'secret',
    'refresh_token_url': 'refresh_token_url',
    'scope': 'default_scope',
    'token_url': 'token_url',
    'yandex_client_id': 'related_yandex_client_id',
}


class _BaseStationApplicationView(InternalApiHandlerV3):
    def _fill_application_from_form_values(self, application):
        for form_name, value in self.form_values.iteritems():
            if value is not Undefined:
                model_name = _FORM_NAME_TO_MODEL_NAME[form_name]
                setattr(application, model_name, value)

    def _build_application_response(self, app):
        app_response = dict()
        for form_name, model_name in _FORM_NAME_TO_MODEL_NAME.iteritems():
            app_response[form_name] = getattr(app, model_name)
        app_response = remove_undefined_values(remove_none_values(app_response))
        return self._mask_application_response(app_response)

    def _mask_application_response(self, response):
        if 'client_secret' in response:
            response['masked_client_secret'] = MASK_PLACEHOLDER
            del response['client_secret']
        return response


class _CreateStationApplicationForm(validators.Schema):
    application_display_name = validators.ApplicationDisplayName(if_missing=None)
    client_id = validators.ClientId()
    client_secret = validators.ClientSecret()
    authorization_url = validators.AuthorizationUrl()
    token_url = validators.TokenUrl()
    refresh_token_url = validators.TokenUrl(if_missing=None)
    scope = validators.ScopeString(if_missing=None)
    yandex_client_id = validators.ClientId(if_missing=None)


class CreateStationApplication(_BaseStationApplicationView):
    required_grants = ['create-station-application']
    basic_form = _CreateStationApplicationForm()

    def _process_request(self):
        self._check_rate_limit()

        app_name = build_application_name()
        app = Application(
            name=app_name,
            group_id=ApplicationGroupId.station,
            is_third_party=True,
            domain=_STATION_DEFAULT_DOMAIN,
        )
        self._fill_application_from_form_values(app)
        self._session.add(app)
        self._session.commit()

        self._update_rate_counter()

        self.response_values.update(
            dict(
                application=self._build_application_response(app),
            ),
        )

    def _check_rate_limit(self):
        counter_desc = CreateApplicationByIpCounterDescriptor(self._user_ip)
        try:
            counters = self._kolmogor.get(SOCIALISM_KOLMOGOR_NAMESPACE, [counter_desc.name])
        except (KolmogorTemporaryError, KolmogorPermanentError):
            raise KolmogorFailedWebServiceError()
        if counters[counter_desc.name] >= counter_desc.limit:
            raise RateLimitExceededWebServiceError()

    def _update_rate_counter(self):
        counter_desc = CreateApplicationByIpCounterDescriptor(self._user_ip)
        try:
            self._kolmogor.inc(SOCIALISM_KOLMOGOR_NAMESPACE, [counter_desc.name])
        except (KolmogorTemporaryError, KolmogorPermanentError):
            logger.warning('Failed to update counter: %s' % counter_desc.name)


class _RemoveStationApplicationForm(validators.Schema):
    application_name = validators.ApplicationName()


class RemoveStationApplication(_BaseStationApplicationView):
    required_grants = ['remove-station-application']
    basic_form = _RemoveStationApplicationForm()

    def _process_request(self):
        app = providers.get_application_by_name(self.form_values['application_name'])
        if not app:
            return

        if app.group_id != ApplicationGroupId.station:
            return

        self._session.add_committed(app)
        self._session.remove(app)
        self._session.commit()


class _ChangeStationApplicationForm(validators.Schema):
    application_display_name = validators.ApplicationDisplayName(if_missing=Undefined)
    application_name = validators.ApplicationName()
    client_id = validators.ClientId(if_missing=Undefined)
    client_secret = validators.ClientSecret(if_missing=Undefined)
    authorization_url = validators.AuthorizationUrl(if_missing=Undefined)
    token_url = validators.TokenUrl(if_missing=Undefined)
    refresh_token_url = validators.TokenUrl(if_missing=Undefined, not_empty=False, if_empty=None)
    scope = validators.ScopeString(if_missing=Undefined, not_empty=False, if_empty=None)
    yandex_client_id = validators.ClientId(if_missing=Undefined, not_empty=False, if_empty=None)


class ChangeStationApplication(_BaseStationApplicationView):
    required_grants = ['change-station-application']
    basic_form = _ChangeStationApplicationForm()

    def _process_request(self):
        app = providers.get_application_by_name(self.form_values['application_name'])
        if not app:
            raise ApplicationUnknownWebServiceError()

        if app.group_id != ApplicationGroupId.station:
            raise ApplicationUnknownWebServiceError()

        self._session.add_committed(app)
        self._fill_application_from_form_values(app)
        self._session.commit()

        self.response_values.update(
            dict(
                application=self._build_application_response(app),
            ),
        )


class _GetStationApplicationsForm(validators.Schema):
    application_names = validators.ApplicationNameSet()


class GetStationApplications(_BaseStationApplicationView):
    required_grants = ['get-station-applications']
    basic_form = _GetStationApplicationsForm()

    def _process_request(self):
        apps, unknown_names = providers.get_many_applications_by_names(self.form_values['application_names'])

        station_apps = list()
        for app in apps:
            if app.group_id == ApplicationGroupId.station:
                station_apps.append(app)
            else:
                unknown_names.append(app.name)

        app_name_to_response = dict()

        for unknown_app_name in unknown_names:
            app_name_to_response[unknown_app_name] = None

        for app in station_apps:
            app_response = self._build_application_response(app)
            app_name_to_response[app.name] = app_response

        self.response_values.update(dict(applications=app_name_to_response))
