# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.social.api.common import (
    execute,
    executew,
    invalidate_billing_cache,
)
from passport.backend.social.api.statbox import StatboxLogger
from passport.backend.social.api.views.v3.base import InternalApiHandlerV3
from passport.backend.social.common import validators
from passport.backend.social.common.builders.billing import BillingApiError
from passport.backend.social.common.builders.passport import (
    PassportAccountNotFoundError,
    PassportPermanentError,
    PassportTemporaryError,
)
from passport.backend.social.common.db.schemas import (
    person_table,
    profile_table,
)
from passport.backend.social.common.misc import is_account_like_portalish
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_logging import (
    BindingLogger,
    BindingsDeletedStatboxEvent,
)
from passport.backend.social.common.web_service import (
    AccountInvalidTypeWebServiceError,
    AccountNotFoundWebServiceError,
    BillingApiFailedWebServiceError,
    PassportFailedWebServiceError,
)
from sqlalchemy import sql


class _UnbindPhonishByUidForm(validators.Schema):
    uid1 = validators.Userid()
    uid2 = validators.Userid()
    delete_phone = validators.StringBool(if_missing=False, not_empty=True)
    logout_master = validators.StringBool(if_missing=False, not_empty=True)
    logout_phonish = validators.StringBool(if_missing=False, not_empty=True)


class UnbindPhonishByUid(InternalApiHandlerV3):
    basic_form = _UnbindPhonishByUidForm
    required_grants = ['unbind-phonish-by-uid']

    def _process_request(self):
        accounts = self._get_accounts_from_uids()
        if accounts[0].is_phonish:
            portal_account, phonish_account = accounts[1], accounts[0]
        else:
            portal_account, phonish_account = accounts[0], accounts[1]

        if not is_account_like_portalish(portal_account):
            e = AccountInvalidTypeWebServiceError()
            e.description = 'One of uids should be kind of portal account'
            raise e
        if not phonish_account.is_phonish:
            e = AccountInvalidTypeWebServiceError()
            e.description = 'One of uids should be phonish account'
            raise e

        profile_id = self._find_profile(portal_account, phonish_account)

        if profile_id is not None:
            if self.form_values['delete_phone']:
                self._delete_phonish_phone_from_portal_account(portal_account, phonish_account)
            if self.form_values['logout_master']:
                self._logout_account(portal_account)
            if self.form_values['logout_phonish']:
                self._logout_account(phonish_account)
            self._delete_profile(profile_id)

        # Инвалидируем кеш Билли всегда, чтобы можно было повторить даже, если
        # случился сбой.
        self._invalidate_billing_cache(portal_account.uid, phonish_account.uid)

    def _get_accounts_from_uids(self):
        # Get_many_accounts_by_uids падает, если в него передавать одинаковые
        # uid'ы, поэтому нужно подготовить список uid'ов правильно.
        uids = [self.form_values['uid1']]
        if self.form_values['uid2'] != self.form_values['uid1']:
            uids.append(self.form_values['uid2'])
        accounts, unknown_uids = self._account_getter.get_many_accounts_from_uids(uids)
        if unknown_uids:
            e = AccountNotFoundWebServiceError()
            e.description = 'Accounts for uids %s are not found' % ', '.join(map(str, unknown_uids))
            raise e
        if self.form_values['uid2'] == self.form_values['uid1']:
            accounts.append(accounts[0].snapshot())
        return accounts

    def _find_profile(self, portal_account, phonish_account):
        query = (
            sql.select([profile_table])
            .where(
                sql.and_(
                    profile_table.c.uid == portal_account.uid,
                    profile_table.c.provider_id == Yandex.id,
                    profile_table.c.userid == str(phonish_account.uid),
                ),
            )
        )
        profile = execute(query).fetchone()
        if profile:
            return profile.profile_id

    def _delete_profile(self, profile_id):
        executew(
            sql.delete(profile_table)
            .where(profile_table.c.profile_id == profile_id),
        )
        executew(
            sql.delete(person_table)
            .where(person_table.c.profile_id == profile_id),
        )

    _DROP_PHONE_OK_STATUS_SET = {'OK', 'NOTFOUND'}

    def _delete_phonish_phone_from_portal_account(self, portal_account, phonish_account):
        phone_number = phonish_account.phones.default.number
        phone = portal_account.phones.by_number(phone_number)
        if not (phone and phone != portal_account.phones.secure):
            return
        try:
            response = self._passport_api.drop_phone(portal_account.uid, phone.id)
        except (
            PassportPermanentError,
            PassportTemporaryError,
        ):
            raise PassportFailedWebServiceError()
        if response['status'].upper() not in self._DROP_PHONE_OK_STATUS_SET:
            raise NotImplementedError()

    def _logout_account(self, account):
        try:
            self._passport_api.account_options(account.uid, global_logout=True)
        except PassportAccountNotFoundError:
            pass
        except (
            PassportPermanentError,
            PassportTemporaryError,
        ):
            raise PassportFailedWebServiceError()

    def _invalidate_billing_cache(self, portal_uid, phonish_uid):
        StatboxLogger().log(
            action='update_account_yandex_bindings',
            uid=portal_uid,
            consumer=self._consumer,
        )
        BindingLogger().log_event(
            BindingsDeletedStatboxEvent(
                master_uid=portal_uid,
                slave_provider_userids=[(Yandex.code, phonish_uid)],
            ),
        )
        try:
            invalidate_billing_cache(portal_uid, fail_safe=False)
        except BillingApiError:
            raise BillingApiFailedWebServiceError()
