# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from operator import itemgetter

import flask
from passport.backend.social.api.common import execute
from passport.backend.social.api.views.v3.base import InternalApiHandlerV3
from passport.backend.social.common import validators
from passport.backend.social.common.datastructures import DirectedGraph
from passport.backend.social.common.db.schemas import profile_table
from passport.backend.social.common.limits import get_qlimits
from passport.backend.social.common.misc import is_account_like_portalish
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.providers.Kinopoisk import Kinopoisk
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.web_service import (
    AccountGetter,
    AccountNotFoundWebServiceError,
)
from passport.backend.utils.time import datetime_to_integer_unixtime
from sqlalchemy.sql.expression import (
    and_ as sql_and,
    not_ as sql_not,
    select as sql_select,
)


class _BaseWhoSharesForm(validators.Schema):
    provider = validators.ProviderCode()
    userid = validators.Userid()


class _BaseWhoSharesView(InternalApiHandlerV3):
    def _response_success(self):
        self._cast_userids_in_response_to_strings()
        self._sort_user_lists_in_response()
        return super(_BaseWhoSharesView, self)._response_success()

    def _cast_userids_in_response_to_strings(self):
        for provider, accounts in self.response_values['accounts'].iteritems():
            self.response_values['accounts'][provider] = map(str, accounts)

    def _sort_user_lists_in_response(self):
        for provider in self.response_values['accounts']:
            self.response_values['accounts'][provider].sort()

    def _response_fail(self, *args, **kwargs):
        self.response_values.pop('accounts', None)
        return super(_BaseWhoSharesView, self)._response_fail(*args, **kwargs)


class _WhoSharesPaymentCardsForm(_BaseWhoSharesForm):
    pass


class WhoSharesPaymentCards(_BaseWhoSharesView):
    required_grants = ['who-shares-payment-cards']
    basic_form = _WhoSharesPaymentCardsForm()

    def _process_request(self):
        self.response_values.update(
            accounts={
                Kinopoisk.code: [],
                Yandex.code: [],
            },
        )

        provider_code = self.form_values['provider']

        if provider_code not in {Kinopoisk.code, Yandex.code}:
            return

        try:
            subject_account = self._get_account_from_uid(self.form_values['userid'])
        except AccountNotFoundWebServiceError:
            return

        if provider_code == Yandex.code and subject_account.is_kinopoisk:
            return

        if provider_code == Kinopoisk.code and not subject_account.is_kinopoisk:
            return

        if is_account_like_portalish(subject_account):
            provider_code_to_userids = self._who_shares_payment_cards_with_portalish(
                self.form_values['userid'],
            )
        elif subject_account.is_kinopoisk:
            provider_code_to_userids = self._who_shares_payment_cards_with_kinopoisk(
                self.form_values['userid'],
            )
        else:
            return

        for provider_code in provider_code_to_userids:
            provider_user_ids = provider_code_to_userids.get(provider_code, [])
            self.response_values['accounts'][provider_code] = provider_user_ids

    def _who_shares_payment_cards_with_portalish(self, userid):
        federation = Federation(Yandex.code, userid)
        federation.load_children()
        federation.filter_by_provider_codes(
            [
                Kinopoisk.code,
                Yandex.code,
            ],
        )
        federation.limit(social_config.max_federation_size)
        federation.load_yandex_accounts()

        def _filter_yandex_account(account, relation):
            return account.is_phonish
        federation.filter_yandex_accounts(_filter_yandex_account)

        return federation.get_userids()

    def _who_shares_payment_cards_with_kinopoisk(self, userid):
        federation = Federation(Kinopoisk.code, userid)
        federation.load_parents_and_siblings()
        federation.filter_by_provider_codes(
            [
                Kinopoisk.code,
                Yandex.code,
            ],
        )
        federation.limit(social_config.max_federation_size)

        def _filter_by_provider_userid_relation(provider_code, userid, relation):
            return (
                provider_code == Yandex.code
                or
                relation == _Relation.sibling and provider_code == Kinopoisk.code
            )
        federation.filter_by_provider_userid_relation(_filter_by_provider_userid_relation)

        federation.load_yandex_accounts()

        def _filter_yandex_account(account, relation):
            return (
                relation == _Relation.parent and is_account_like_portalish(account)
                or
                relation == _Relation.sibling and account.is_phonish
            )
        federation.filter_yandex_accounts(_filter_yandex_account)

        return federation.get_userids()


class _WhoSharesTaxiDataForm(_BaseWhoSharesForm):
    pass


class WhoSharesTaxiData(_BaseWhoSharesView):
    required_grants = ['who-shares-taxi-data']
    basic_form = _WhoSharesTaxiDataForm()

    def _process_request(self):
        self.response_values.update(accounts={Yandex.code: []})

        provider_code = self.form_values['provider']

        if provider_code != Yandex.code:
            return

        try:
            subject_account = self._get_account_from_uid(self.form_values['userid'])
        except AccountNotFoundWebServiceError:
            return

        if subject_account.is_phonish:
            provider_code_to_userids = self._who_shares_taxi_data_with_phonish(
                self.form_values['userid'],
            )
        elif is_account_like_portalish(subject_account):
            provider_code_to_userids = self._who_shares_taxi_data_with_portalish(
                self.form_values['userid'],
            )
        else:
            return

        self.response_values['accounts'][Yandex.code] = provider_code_to_userids.get(Yandex.code, [])

    def _who_shares_taxi_data_with_phonish(self, userid):
        federation = Federation(Yandex.code, userid)
        federation.load_parents_and_siblings()
        federation.filter_by_provider_codes([Yandex.code])
        federation.limit(social_config.max_federation_size)
        federation.load_yandex_accounts()

        def _filter_yandex_account(account, relation):
            return (
                relation == _Relation.parent and is_account_like_portalish(account)
                or
                relation == _Relation.sibling and account.is_phonish
            )

        federation.filter_yandex_accounts(_filter_yandex_account)

        return federation.get_userids()

    def _who_shares_taxi_data_with_portalish(self, userid):
        federation = Federation(Yandex.code, userid)
        federation.load_children()
        federation.filter_by_provider_codes([Yandex.code])
        federation.limit(social_config.max_federation_size)
        federation.load_yandex_accounts()

        def _filter_yandex_account(account, relation):
            return relation == _Relation.child and account.is_phonish
        federation.filter_yandex_accounts(_filter_yandex_account)

        return federation.get_userids()


class WhoSharesTaxiDataV2(_BaseWhoSharesView):
    required_grants = ['who-shares-taxi-data-v2']
    basic_form = _WhoSharesTaxiDataForm()

    def _process_request(self):
        self.response_values.update(accounts={Yandex.code: []})

        provider_code = self.form_values['provider']

        if provider_code != Yandex.code:
            return

        try:
            subject_account = self._get_account_from_uid(self.form_values['userid'])
        except AccountNotFoundWebServiceError:
            return

        if not is_account_like_portalish(subject_account):
            return

        federation = Federation(provider_code, self.form_values['userid'])
        federation.load_children()
        federation.filter_by_provider_codes([Yandex.code])
        federation.limit(social_config.max_federation_size)
        federation.load_yandex_accounts()

        def _filter_yandex_account(account, relation):
            return relation == _Relation.child and account.is_phonish
        federation.filter_yandex_accounts(_filter_yandex_account)

        provider_code_to_userids = federation.get_userids()

        yandex_response = []
        yandex_userids = provider_code_to_userids.get(Yandex.code, [])
        for userid in yandex_userids:
            account = federation.get_loaded_yandex_account(userid)
            yandex_response.append(
                {
                    'userid': userid,
                    'phones': [
                        {
                            'number': p.number.e164,
                            'confirmed': datetime_to_integer_unixtime(p.confirmed),
                        }
                        for p in account.phones.bound().itervalues()
                    ],
                },
            )

        self.response_values['accounts'][Yandex.code] = yandex_response

    def _cast_userids_in_response_to_strings(self):
        for provider, accounts in self.response_values['accounts'].iteritems():
            processed_accounts = []
            for account in accounts:
                processed_accounts.append(
                    dict(account, userid=str(account['userid'])),
                )
            self.response_values['accounts'][provider] = processed_accounts

    def _sort_user_lists_in_response(self):
        for provider in self.response_values['accounts']:
            self.response_values['accounts'][provider].sort(key=itemgetter('userid'))


class Federation(object):
    def __init__(self, provider_code, userid):
        self._g = DirectedGraph()
        self._citizen_node = dict()
        self._subject_citizen = _Citizen(provider_code=provider_code, userid=userid)
        self._subject_node = self._g.add_node(self._subject_citizen)
        self._add_citizen_node(self._subject_citizen, self._subject_node)
        self._userid_to_yandex_account = dict()

    def _traverse_g(self):
        for node, _ in self._traverse_g_with_path():
            yield node

    def _traverse_g_with_path(self):
        for node, path in self._g.traverse_bfs(
            start=self._subject_node,
            edge_key=lambda e: -e.value.profile_id,
        ):
            if node is not self._subject_node:
                yield (node, path)

    def _add_citizen_node(self, citizen, node):
        self._citizen_node[citizen] = node

    def _remove_citizen_node(self, citizen):
        self._citizen_node.pop(citizen)

    def _find_node_by_citizen(self, citizen):
        return self._citizen_node.get(citizen)

    def _get_relation_type_from_path(self, path):
        if not path:
            # Считаем что петли невозможны
            assert False  # pragma: no cover
        return path[-1].value.relation_type

    def _get_siblings(self, parents):
        assert all(a.provider_code == Yandex.code for a, _ in parents)

        subject_citizen_provider_info = providers.get_provider_info_by_name(
            self._subject_citizen.provider_code,
        )
        subject_citizen_provider_id = subject_citizen_provider_info['id']
        query = (
            sql_select([
                profile_table.c.provider_id,
                profile_table.c.profile_id,
                profile_table.c.userid,
                profile_table.c.uid,
            ])
            .where(
                sql_and(
                    profile_table.c.uid.in_([int(a.userid) for a, _ in parents]),
                    sql_not(
                        sql_and(
                            profile_table.c.provider_id == subject_citizen_provider_id,
                            profile_table.c.userid == self._subject_citizen.userid,
                        ),
                    ),
                ),
            )
            .order_by(profile_table.c.profile_id.desc())
            .limit(get_qlimits()['profiles'])
        )
        profiles = execute(query).fetchall()

        siblings = []
        uid_to_parent = dict((int(c.userid), c) for c, _ in parents)
        for profile in profiles:
            provider_info = providers.get_provider_info_by_id(profile.provider_id)
            if not provider_info:
                # Пропускаем профили удалённых провайдеров
                continue
            citizen = _Citizen(
                provider_code=provider_info['code'],
                userid=profile.userid,
            )
            relation = _Relation(profile_id=profile.profile_id, relation_type=_Relation.sibling)
            parent = uid_to_parent[profile.uid]
            siblings.append(((citizen, relation), parent))
        return siblings

    def _get_many_accounts_by_uids(self, uids):
        return self._account_getter.get_many_accounts_from_uids(sorted(uids))

    @property
    def _account_getter(self):
        return AccountGetter(http_pool_manager=get_http_pool_manager(), request=flask.request)

    def load_children(self):
        children = self.get_children()
        for citizen, relation in children:
            node = self._find_node_by_citizen(citizen)
            if node is None:
                node = self._g.add_node(citizen)
                self._add_citizen_node(citizen, node)
            self._g.add_edge(src=self._subject_node, dst=node, value=relation)

    def get_children(self):
        assert self._subject_citizen.provider_code == Yandex.code

        query = (
            sql_select([
                profile_table.c.provider_id,
                profile_table.c.profile_id,
                profile_table.c.userid,
            ])
            .where(profile_table.c.uid == int(self._subject_citizen.userid))
            .order_by(profile_table.c.profile_id.desc())
            .limit(get_qlimits()['profiles'])
        )
        profiles = execute(query).fetchall()

        children = []
        for profile in profiles:
            provider_info = providers.get_provider_info_by_id(profile.provider_id)
            if not provider_info:
                # Пропускаем профили удалённых провайдеров
                continue
            citizen = _Citizen(
                provider_code=provider_info['code'],
                userid=profile.userid,
            )
            relation = _Relation(profile_id=profile.profile_id, relation_type=_Relation.child)
            children.append((citizen, relation))
        return children

    def filter_by_provider_codes(self, filter_codes):
        for node in self._traverse_g():
            citizen = node.value
            if citizen.provider_code not in filter_codes:
                self._remove_citizen_node(citizen)
                self._g.remove_node(node)

    def limit(self, max_size):
        traversed_size = 1
        for node in self._traverse_g():
            if traversed_size > max_size:
                citizen = node.value
                self._remove_citizen_node(citizen)
                self._g.remove_node(node)
            traversed_size += 1

    def load_yandex_accounts(self):
        yandex_citizens = []
        for node in self._traverse_g():
            citizen = node.value
            if citizen.provider_code == Yandex.code:
                yandex_citizens.append(citizen)

        missed_accounts = []
        for citizen in yandex_citizens:
            if citizen.userid not in self._userid_to_yandex_account:
                missed_accounts.append(citizen.userid)

        fetched_accounts, removed_uids = self._get_many_accounts_by_uids(missed_accounts)
        for account in fetched_accounts:
            self._userid_to_yandex_account[str(account.uid)] = account

    def get_loaded_yandex_account(self, userid):
        return self._userid_to_yandex_account[str(userid)]

    def filter_yandex_accounts(self, filter_function):
        def _filter_yandex_account_function(provider_code, userid, relation_type):
            if provider_code != Yandex.code:
                return True
            account = self._userid_to_yandex_account.get(userid)
            return account is not None and filter_function(account, relation_type)
        return self.filter_by_provider_userid_relation(_filter_yandex_account_function)

    def filter_by_provider_userid_relation(self, filter_function):
        nodes_and_paths = []
        for node, path in self._traverse_g_with_path():
            citizen = node.value
            nodes_and_paths.append((node, path))

        for node, path in nodes_and_paths:
            relation_type = self._get_relation_type_from_path(path)
            citizen = node.value
            if not filter_function(citizen.provider_code, citizen.userid, relation_type):
                self._remove_citizen_node(citizen)
                self._g.remove_node(node)

    def get_userids(self):
        userids = dict()
        for node in self._traverse_g():
            citizen = node.value
            provider = userids.setdefault(citizen.provider_code, set())
            provider.add(citizen.userid)
        for provider_code in userids:
            userids[provider_code] = list(userids[provider_code])
        return userids

    def load_parents_and_siblings(self):
        parents, siblings_and_parents = self.get_parents_and_siblings()

        for citizen, relation in parents:
            node = self._find_node_by_citizen(citizen)
            if node is None:
                node = self._g.add_node(citizen)
                self._add_citizen_node(citizen, node)
            self._g.add_edge(src=self._subject_node, dst=node, value=relation)

        for (citizen, relation), parent_citizen in siblings_and_parents:
            node = self._find_node_by_citizen(citizen)
            if node is None:
                node = self._g.add_node(citizen)
                self._add_citizen_node(citizen, node)
            parent_node = self._find_node_by_citizen(parent_citizen)
            self._g.add_edge(src=parent_node, dst=node, value=relation)

    def get_parents_and_siblings(self):
        parents = self.get_parents()
        siblings_and_parents = self._get_siblings(parents)
        return parents, siblings_and_parents

    def get_parents(self):
        subject_citizen_provider_info = providers.get_provider_info_by_name(
            self._subject_citizen.provider_code,
        )
        subject_citizen_provider_id = subject_citizen_provider_info['id']
        query = (
            sql_select([
                profile_table.c.profile_id,
                profile_table.c.uid,
            ])
            .where(
                sql_and(
                    profile_table.c.userid == self._subject_citizen.userid,
                    profile_table.c.provider_id == subject_citizen_provider_id,
                ),
            )
            .order_by(profile_table.c.profile_id.desc())
            .limit(get_qlimits()['profiles'])
        )
        profiles = execute(query).fetchall()

        parents = []
        for profile in profiles:
            citizen = _Citizen(provider_code=Yandex.code, userid=profile.uid)
            relation = _Relation(profile_id=profile.profile_id, relation_type=_Relation.parent)
            parents.append((citizen, relation))
        return parents


class _Relation(object):
    child = 'child'
    parent = 'parent'
    sibling = 'sibling'
    unknown = 'unknown'

    def __init__(self, profile_id, relation_type):
        self.profile_id = profile_id
        self.relation_type = relation_type


class _Citizen(object):
    def __init__(self, provider_code, userid):
        self.provider_code = provider_code
        self.userid = str(userid)

    def __eq__(self, other):
        if type(other) is not _Citizen:
            return NotImplemented  # pragma: no cover
        return (
            self.provider_code == other.provider_code and
            self.userid == other.userid
        )

    def __ne__(self, other):
        if type(other) is not _Citizen:
            return NotImplemented  # pragma: no cover
        return not self.__eq__(other)

    def __hash__(self):
        key = (self.provider_code, self.userid)
        return hash(key)
