# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import datetime
from functools import partial
import logging
from random import SystemRandom

from flask import (
    Flask,
    request,
)
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.tvm.tvm_credentials_manager import TvmCredentialsManager
from passport.backend.social.broker.communicators.communicator import CommunicatorApplicationMapper
from passport.backend.social.broker.handlers import (
    check_pkce,
    create_task_for_profile,
    ping,
    redirect,
    tw_reverse_auth_token,
)
from passport.backend.social.broker.handlers.profile import (
    authz_in_web as authz_in_web_for_profile,
    bind_by_token as bind_profile_by_token,
    bind_kinopoisk_account,
    bind_phonish_account_by_track,
    bind_phonish_account_by_uid,
    bind_yandex_account,
    does_binding_exist,
    task_by_token as task_with_profile_by_token,
)
from passport.backend.social.broker.handlers.test import create_task as test_create_task
from passport.backend.social.broker.handlers.token import (
    authz_in_app as authz_in_app_for_token,
    authz_in_web as authz_in_web_for_token,
)
from passport.backend.social.broker.logging_settings import logging_settings_init
from passport.backend.social.common import (
    middleware,
    redis_client,
)
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.utils import (
    build_master_db_config,
    build_slave_db_config,
    create_engine,
)
from passport.backend.social.common.grants import GrantsConfig
from passport.backend.social.common.limits import QLimits
from passport.backend.social.common.misc import (
    dump_to_json_string,
    GraphiteMessageType,
    name_for_graph_log,
    write_graph_log_message,
)
from passport.backend.social.common.provider_settings import providers as provider_settings
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.tvm import SocialTvmCredentialsManager
from passport.backend.social.common.useragent import (
    build_http_pool_manager,
    ZoraUseragent,
)
from passport.backend.social.common.web_service import (
    Request,
    Response,
)
import passport.backend.social.proxylib
from passport.backend.utils.gost.jwt import install_gost_to_jws
import werkzeug.exceptions


logger = logging.getLogger(__name__)


def prepare_interprocess_environment():
    """
    Инициализация подсистем, которые могут быть общими для нескольких процессов.

    Здесь следует инициализировать только подсистемы, которые не меняют своего
    состояния на протяжении жизни приложения.
    """
    logging_settings_init()
    social_config.init()

    LazyLoader.register('communicator_application_mapper', CommunicatorApplicationMapper)
    LazyLoader.get_instance('communicator_application_mapper')

    LazyLoader.register('qlimits', QLimits)
    LazyLoader.get_instance('qlimits')

    provider_settings.init()
    passport.backend.social.proxylib.init()

    install_gost_to_jws()


def prepare_intraprocess_environment():
    """
    Инициализация подсистем, которые нельзя делать общими для нескольких
    процессов.

    Здесь следует инициализировать подсистемы, которые меняют своё состояние
    на протяжении жизни приложения.
    """
    LazyLoader.register('chrono', lambda: datetime.datetime)
    LazyLoader.get_instance('chrono')

    LazyLoader.register('randomizer', SystemRandom)
    LazyLoader.get_instance('randomizer')

    LazyLoader.register('http_pool_manager', build_http_pool_manager)
    LazyLoader.get_instance('http_pool_manager')

    mysql_read = create_engine(
        build_slave_db_config(social_config),
        False,
        pool_recycle=social_config.broker_pool_recycle,
    )
    LazyLoader.register('slave_db_engine', lambda: mysql_read)
    LazyLoader.get_instance('slave_db_engine')

    mysql_write = create_engine(
        build_master_db_config(social_config),
        False,
        pool_recycle=social_config.broker_pool_recycle,
    )
    LazyLoader.register('master_db_engine', lambda: mysql_write)
    LazyLoader.get_instance('master_db_engine')

    redis_client.RedisClient.init()
    LazyLoader.register('redis', redis_client.RedisClient)
    LazyLoader.get_instance('redis')

    tvm_credentials_manager = SocialTvmCredentialsManager()
    LazyLoader.register('TvmCredentialsManager', lambda: tvm_credentials_manager)
    LazyLoader.get_instance('TvmCredentialsManager')

    grants_config = GrantsConfig(
        'social-broker',
        backup_tvm_credentials_manager=TvmCredentialsManager(
            cache_time=social_config.broker_tvm_cache_time,
            keyring_config_name=social_config.broker_tvm_keyring_config_name,
            ticket_parser2_blackbox_env=social_config.tvm_environment,
        ),
        tvm_credentials_manager=tvm_credentials_manager,
    )
    LazyLoader.register('grants_config', lambda: grants_config)
    LazyLoader.get_instance('grants_config')

    zora_useragent = ZoraUseragent(tvm_credentials_manager=tvm_credentials_manager)
    LazyLoader.register('zora_useragent', lambda: zora_useragent)
    LazyLoader.get_instance('zora_useragent')


def create_app():
    app = Flask(__name__)
    app.request_class = Request
    app.response_class = Response

    app.add_url_rule(
        '/ping', methods=['GET'],
        view_func=ping.PingHandler.as_view())
    app.add_url_rule(
        '/redirect', methods=['POST'],
        view_func=redirect.RedirectHandler.as_view())
    app.add_url_rule(
        '/check_pkce', methods=['POST'],
        view_func=check_pkce.CheckPkceHandler.as_view())
    app.add_url_rule(
        '/task_by_token', methods=['POST'],
        view_func=task_with_profile_by_token.TaskByTokenHandler.as_view())
    app.add_url_rule(
        '/bind_by_token', methods=['POST'],
        view_func=bind_profile_by_token.BindByTokenHandler.as_view())
    app.add_url_rule(
        '/does_profile_exist_by_token', methods=['GET', 'POST'],
        view_func=does_binding_exist.DoesBindingExistByToken.as_view())
    app.add_url_rule(
        '/bind_yandex_by_token', methods=['POST'],
        view_func=bind_yandex_account.BindYandexAccountByTokenHandler.as_view())
    app.add_url_rule(
        '/bind_kinopoisk_account_by_token', methods=['POST'],
        view_func=bind_kinopoisk_account.BindKinopoiskAccountByTokenHandler.as_view())
    app.add_url_rule(
        '/bind_kinopoisk_account_by_session_id', methods=['POST'],
        view_func=bind_kinopoisk_account.BindKinopoiskAccountBySessionIdHandler.as_view())
    app.add_url_rule(
        '/bind_phonish_account_by_track_v2', methods=['POST'],
        view_func=bind_phonish_account_by_track.BindPhonishAccountByTrackHandlerV2.as_view())
    app.add_url_rule(
        '/bind_phonish_account_by_uid', methods=['POST'],
        view_func=bind_phonish_account_by_uid.BindPhonishAccountByUidHandler.as_view())
    app.add_url_rule(
        '/create_task_for_profile', methods=['POST'],
        view_func=create_task_for_profile.CreateTaskForProfile.as_view())

    app.add_url_rule(
        '/tw_reverse_auth_token', methods=['GET', 'POST'],
        view_func=tw_reverse_auth_token.TwitterReverseAuthTokenHandler.as_view())

    app.add_url_rule(
        '/authz_in_app/start', methods=['POST'],
        view_func=authz_in_app_for_token.AuthzInAppForTokenStartHandler.as_view())
    app.add_url_rule(
        '/authz_in_app/<string:task_id>/callback', methods=['POST'],
        view_func=authz_in_app_for_token.AuthzInAppForTokenCallbackHandler.as_view())
    app.add_url_rule(
        '/authz_in_app/<string:task_id>/continue', methods=['POST'],
        view_func=authz_in_app_for_token.AuthzInAppForTokenContinueHandler.as_view())
    app.add_url_rule(
        '/authz_in_app/entrust_to_account', methods=['POST'],
        view_func=authz_in_app_for_token.AuthzInAppForTokenEntrustToAccountHandler.as_view())

    app.add_url_rule(
        '/authz_in_web/start', methods=['POST'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenStartHandler.as_view())
    app.add_url_rule(
        '/authz_in_web/callback', methods=['POST'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenCallbackHandler.as_view())
    app.add_url_rule(
        '/authz_in_web/bind/submit', methods=['POST'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenBindSubmitHandler.as_view())
    app.add_url_rule(
        '/authz_in_web/bind/commit', methods=['POST'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenBindCommitHandler.as_view())
    app.add_url_rule(
        '/authz_in_web/collect_diagnostics', methods=['POST'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenCollectDiagnosticsHandler.as_view())
    app.add_url_rule(
        '/authz_in_web/read_diagnostics', methods=['GET'],
        view_func=authz_in_web_for_token.AuthzInWebForTokenReadDiagnosticsHandler.as_view())

    app.add_url_rule(
        '/test/create_task', methods=['POST'],
        view_func=test_create_task.TestCreateTaskHandler.as_view())

    app.add_url_rule(
        '/start', methods=['POST'],
        view_func=authz_in_web_for_profile.StartHandler.as_view())
    app.add_url_rule(
        '/<string:task_id>/retry', methods=['POST'],
        view_func=authz_in_web_for_profile.RetryHandler.as_view())
    app.add_url_rule(
        '/<string:task_id>/callback', methods=['POST'],
        view_func=authz_in_web_for_profile.CallbackHandler.as_view())
    app.add_url_rule(
        '/<string:task_id>/continue', methods=['POST'],
        view_func=authz_in_web_for_profile.ContinueHandler.as_view())
    app.add_url_rule(
        '/<string:task_id>/bind', methods=['POST'],
        view_func=authz_in_web_for_profile.BindHandler.as_view())

    app.before_request(middleware.setup_request)
    app.before_request(middleware.log_http_request)
    app.before_request(middleware.log_api_call)
    app.after_request(partial(middleware.after_request, 'social-broker-access-log'))
    app.errorhandler(Exception)(handle_exception)
    app.errorhandler(404)(handle_page_not_found)
    app.errorhandler(werkzeug.exceptions.MethodNotAllowed)(handle_page_not_found)

    return app


def _build_error_response(api_error_code='InternalError', message='', status=500):
    response = dump_to_json_string({
        'error': {
            'code': api_error_code,
            'message': message,
        },
        'request_id': request.id,
    })
    response = Response(response, status=status, mimetype='application/json')
    response.api_status = 'error'
    response.api_error_code = api_error_code
    return response


def handle_page_not_found(_):
    return _build_error_response(
        api_error_code='PageNotFound',
        message='Page not found',
        status=404,
    )


def handle_exception(_):
    logger.error('Unhandled exception', exc_info=True)
    app = getattr(request_ctx, 'application', None)
    app_code = name_for_graph_log(app)
    write_graph_log_message(GraphiteMessageType.error, request.endpoint, app_code, 'internal')

    return _build_error_response()
