# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import logging

from passport.backend.social.broker.communicators.communicator import OAuth2CommunicatorWithAbsoluteRedirect
from passport.backend.social.common import oauth2
from passport.backend.social.common.exception import ResponseDecodeProxylibError
from passport.backend.social.common.providers.DnevnikRu import DnevnikRu
from passport.backend.social.common.social_config import social_config


logger = logging.getLogger('social.broker.communicators')


class DnevnikRuCommunicator(OAuth2CommunicatorWithAbsoluteRedirect):
    default_scopes = ['CommonInfo']
    scope_delimiter = ','
    _force_prompt_args = {'prompt': 'login'}
    RESULT_TRANSLATION = {
        'expiresIn': 'expires_in',
        'accessToken': 'access_token',
        'refreshToken': 'refresh_token',
    }

    @property
    def OAUTH_AUTHORIZE_URL(self):
        return social_config.dnevnik_ru_oauth_authorize_url
    REDIRECT_NEEDED = True

    @property
    def OAUTH_ACCESS_TOKEN_URL(self):
        return social_config.dnevnik_ru_oauth_token_url
    ACCESS_TOKEN_REQUEST_TYPE = 'POST'

    provider_code = DnevnikRu.code

    def _build_redirect_args(self, app, callback_url):
        return {
            'redirect_uri': social_config.dnevnik_ru_broker_redirect_uri,
            'state': callback_url,
        }

    def _parse_access_token_response(self, http_response):
        if http_response.status // 100 == 4:
            try:
                decoded = json.loads(http_response.decoded_data)
            except ValueError:
                raise ResponseDecodeProxylibError(
                    'Response is not in a json format: %s' % http_response.decoded_data,
                )
            if 'type' in decoded and 'error' not in decoded:
                decoded['error'] = decoded['type']
            http_response.decoded_data = json.dumps(decoded)
        return super(DnevnikRuCommunicator, self)._parse_access_token_response(http_response)

    def _get_error_detector_for_access_token_response(self):
        return oauth2.token.detect_error

    def parse_access_token(self, response):
        try:
            decoded = json.loads(response)
        except ValueError:
            raise ResponseDecodeProxylibError(
                'Response is not in a json format: %s' % response,
            )

        for _from, _to in self.RESULT_TRANSLATION.items():
            if _from in decoded:
                decoded[_to] = decoded[_from]

        response = json.dumps(decoded)
        return super(DnevnikRuCommunicator, self).parse_access_token(response)
