# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import copy

from passport.backend.social.broker import exceptions
from passport.backend.social.broker.communicators.communicator import OAuth2Communicator
from passport.backend.social.common.exception import InvalidTokenProxylibError
from passport.backend.social.common.providers.Esia import Esia
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import Url
from passport.backend.social.proxylib.EsiaProxy import (
    EsiaPermissions,
    EsiaRefreshToken,
    EsiaToken,
    get_esia_signature,
    get_esia_timestamp,
    parse_and_verify_esia_id_token,
    parse_esia_token,
)
from passport.backend.utils.string import smart_text


class EsiaCommunicator(OAuth2Communicator):
    default_scopes = ['openid']
    IS_OPAQUE_STATE = True
    OAUTH_ACCESS_TOKEN_URL = social_config.property('esia_token_url')
    OAUTH_AUTHORIZE_URL = social_config.property('esia_authorize_url')
    OAUTH_AUTHORIZE_MANDATORY_ARGUMENTS = {'access_type': 'offline'}
    provider_code = Esia.code
    scope_delimiter = ' '

    def _get_authorize_query(self, *args, **kwargs):
        args = super(EsiaCommunicator, self)._get_authorize_query(*args, **kwargs)
        args['timestamp'] = get_esia_timestamp()

        # Сперва построим по всем запрошенным скоупам значение для permissions
        permissions = EsiaPermissions.from_params(args)
        if permissions:
            args['permissions'] = str(permissions)
        # В параметре scope можно передавать только openid, все остальные
        # скоупы должны передаваться в параметре permissions
        args['scope'] = 'openid'

        args['client_secret'] = get_esia_signature(args)
        return args

    def parse_access_token(self, *args, **kwargs):
        token_dict = super(EsiaCommunicator, self).parse_access_token(*args, **kwargs)
        token_dict = copy.copy(token_dict)

        token_dict['value'] = EsiaToken(
            access_token=token_dict.get('value', ''),
            id_token=token_dict.get('id_token', ''),
            scope=self.get_esia_scope_string(),
            version=1,
        ).serialize()

        if token_dict.get('refresh'):
            token_dict['refresh'] = EsiaRefreshToken(
                id_token=token_dict.get('id_token', ''),
                scope=self.get_esia_scope_string(),
                state=self.get_esia_state(),
                value=token_dict['refresh'],
                version=1,
            ).serialize()

        return token_dict

    def sanitize_server_token(self, server_token, refresh_token):
        token_dict = super(EsiaCommunicator, self).sanitize_server_token(server_token, refresh_token)
        try:
            esia_token = parse_esia_token(token_dict.get('value'))
            parse_and_verify_esia_id_token(
                esia_token.id_token,
                verify_signature=social_config.esia_verify_id_token_signature,
            )
        except InvalidTokenProxylibError as e:
            raise exceptions.CommunicationFailedError(str(e))
        return token_dict

    def get_access_token_request(self, *args, **kwargs):
        kwargs.update(callback_url=self.get_task().callback_url)
        url, data, headers = super(EsiaCommunicator, self).get_access_token_request(*args, **kwargs)

        data = copy.copy(data)

        url = Url(url)
        data.update(url.params)
        url = url.paramless

        data.update(
            scope='openid',
            state=self.get_esia_state(),
            timestamp=get_esia_timestamp(),
        )

        data['client_secret'] = get_esia_signature(data)

        return url, data, headers

    def get_esia_scope_string(self):
        task = self.get_task()
        return task.scope if task.scope else self.get_scope()

    def get_esia_state(self):
        return smart_text(self._get_state_from_callback_url(self.get_task().callback_url))
