# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.broker.communicators.communicator import OAuth2CommunicatorWithAbsoluteRedirect
from passport.backend.social.broker.exceptions import (
    CommunicationFailedError,
    UserDeniedError,
)
from passport.backend.social.common import oauth2


logger = logging.getLogger('social.broker.communicators')

INVALID_GRANT_ERROR_CODE = 100


def _detect_error_in_access_token_response(response):
    if 'error' in response:
        error_code = response['error']['code']
        if error_code == INVALID_GRANT_ERROR_CODE:
            raise oauth2.token.InvalidGrant()
        else:
            # Чтобы выяснить какие коды отказа возвращает Facebook, будем
            # логировать все неизвестные.
            logger.error('Failed to exchange authorization code to token: %s' % error_code)
            raise oauth2.token.InvalidGrant()


class FacebookCommunicator(OAuth2CommunicatorWithAbsoluteRedirect):
    OAUTH_ACCESS_TOKEN_URL = 'https://graph.facebook.com/v10.0/oauth/access_token'
    OAUTH_AUTHORIZE_URL = 'https://www.facebook.com/v10.0/dialog/oauth'
    default_scopes = ['user_gender', 'user_link', 'user_birthday', 'email']
    provider_code = 'fb'
    REDIRECT_NEEDED = True

    display_map = dict([(x, x) for x in OAuth2CommunicatorWithAbsoluteRedirect.display_map])
    display_value_if_not_set = 'page'

    _force_prompt_args = {'auth_type': 'reauthorize'}

    def has_error_in_callback(self, query, *args, **kwargs):
        error_name = query.get('error')
        if error_name:
            error_reason = query.get('error_reason')
            if error_name == 'access_denied' and error_reason == 'user_denied':
                raise UserDeniedError('User denied access')
            raise CommunicationFailedError('Error in callback: ' + str(query['error']))

    def get_real_token_data(self, token, fail_silent=False, need_client_id=True):
        return self._get_real_token_data(token, fail_silent=fail_silent, need_client_id=need_client_id)

    def check_token(self, token):
        return self._check_token(token)

    def _get_error_detector_for_access_token_response(self):
        return _detect_error_in_access_token_response
