# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import hashlib
import json
import logging

from passport.backend.social.broker.communicators.communicator import OAuth2Communicator
from passport.backend.social.broker.exceptions import CommunicationFailedError
from passport.backend.social.common.useragent import Url


logger = logging.getLogger(__name__)

INVALID_GRANT_ERROR_CODE = 4


class LastFmCommunicator(OAuth2Communicator):
    OAUTH_ACCESS_TOKEN_URL = 'https://ws.audioscrobbler.com/2.0'
    OAUTH_AUTHORIZE_URL = 'https://www.last.fm/api/auth'
    provider_code = 'lf'

    def parse_access_token(self, response):
        """
        Типичный ответ: {"session":{"name":"Name","key":"60ff810ecadfe2f151afba9e83bd60f8","subscriber":"0"}}
        """
        raw_response = response
        try:
            response = json.loads(raw_response)
            self._check_error_in_access_token_response(response, raw_response)
            return {
                'value': response['session']['key'],
                'expires': None,
            }
        except (KeyError, ValueError):
            self._on_parse_access_token_parsing_error(raw_response)

    def _check_error_in_access_token_response(self, response, raw_response):
        if 'error' in response:
            error_code = response['error']
            if error_code == INVALID_GRANT_ERROR_CODE:
                self._on_parse_access_token_invalid_grant_error(raw_response)
            else:
                logger.error('Failed to exchange authorization code to token: %s' % error_code)
                self._on_parse_access_token_invalid_grant_error(raw_response)

    def _get_authorize_query(self, options):
        query = super(LastFmCommunicator, self)._get_authorize_query(options)
        query['api_key'] = query.pop('client_id')
        query['cb'] = query.pop('redirect_uri')
        del query['response_type']
        return query

    def get_access_token_request(
        self,
        callback_url=None,
        client_id=None,
        client_secret=None,
        code=None,
        scopes=None,
    ):
        logger.debug('[communicator] Getting OAuth2 access token url...')

        args = dict(
            token=code,
            api_key=client_id or self.app.id,
            method='auth.getSession',
            format='json',
        )
        args = self._sign_args(args)

        url = str(Url(self.OAUTH_ACCESS_TOKEN_URL).add_params(args))

        return url, None, None

    def _sign_args(self, args):
        unsignable_keys = {'format'}
        signables = {k: v for k, v in args.iteritems() if k not in unsignable_keys}

        ordered_signables = []
        for key in sorted(signables):
            ordered_signables.append(key)
            ordered_signables.append(signables[key])
        concatenated_signables = ''.join(ordered_signables)

        signature = hashlib.md5(concatenated_signables + self.app.secret).hexdigest()

        signed_args = signables
        signed_args.update(args)
        signed_args['api_sig'] = signature
        return signed_args

    def has_error_in_callback(self, query, *args, **kwargs):
        pass

    def get_exchange_value_from_callback(self, query):
        if 'token' not in query:
            raise CommunicationFailedError('No exchange value specified')
        return query['token']
