# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.broker.communicators.communicator import OAuth2CommunicatorWithAbsoluteRedirect
from passport.backend.social.common import oauth2
from passport.backend.social.common.providers.MosRu import MosRu
from passport.backend.social.common.social_config import social_config


logger = logging.getLogger('social.broker.communicators')


def _detect_error_in_access_token_response(response):
    error_code = response.get('error')
    if error_code:
        if error_code == 'invalid_client':
            raise oauth2.token.InvalidGrant()
    oauth2.token.detect_error(response)


class MosRuCommunicator(OAuth2CommunicatorWithAbsoluteRedirect):
    default_scopes = ['openid', 'profile']
    scope_delimiter = ' '
    OAUTH_AUTH_TYPE_BASIC = True
    _force_prompt_args = {'prompt': 'login'}

    @property
    def OAUTH_AUTHORIZE_URL(self):
        return social_config.mos_ru_oauth_authorize_url
    REDIRECT_NEEDED = True

    @property
    def OAUTH_ACCESS_TOKEN_URL(self):
        return social_config.mos_ru_oauth_token_url
    ACCESS_TOKEN_REQUEST_TYPE = 'POST'

    provider_code = MosRu.code

    def _build_redirect_args(self, app, callback_url):
        return {
            'redirect_uri': social_config.mos_ru_broker_redirect_uri,
            'state': callback_url,
        }

    def _get_error_detector_for_access_token_response(self):
        return _detect_error_in_access_token_response

    def get_scope(self, *args, **kwargs):
        scope = super(MosRuCommunicator, self).get_scope(*args, **kwargs)

        # Мос.ру требует, чтобы openid всегда шёл первым
        if scope:
            scopes = scope.split(self.scope_delimiter)
            try:
                openid_pos = scopes.index('openid')
            except ValueError:
                pass
            else:
                del scopes[openid_pos]
                scope = self.scope_delimiter.join(['openid'] + scopes)
        return scope
