# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from importlib import import_module
import logging
from pkgutil import walk_packages
import uuid

from furl import furl
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.broker.domain import (
    DomainGroup,
    is_domain_mask,
    is_subdomain,
)
from passport.backend.social.broker.error_handler import ErrorHandler
from passport.backend.social.broker.exceptions import (
    AuthorizationErrorResponseCommunicationFailedError,
    BadHttpStatusCommunicationFailedError,
    CommunicationFailedError,
    DisplayInvalidError,
    MissingExchangeValueCommunicationFailedError,
    OAuthTokenInvalidError,
    UserDeniedError,
)
from passport.backend.social.common import oauth2
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import (
    BasicProxylibError,
    FailureSourceType,
    InvalidTokenProxylibError,
    NetworkProxylibError,
    ProviderRateLimitExceededProxylibError,
    ProviderTemporaryUnavailableProxylibError,
    SocialUserDisabledProxylibError,
    UnexpectedResponseProxylibError,
)
from passport.backend.social.common.misc import (
    ApplicationMapper,
    copy,
    expires_in_to_expires_at,
    split_scope_string,
    trim_message,
    urlencode,
    UserParamDescriptor,
)
from passport.backend.social.common.models import Token as TokenModel
from passport.backend.social.common.oauth1 import Oauth1RequestSigner
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.refresh_token.utils import build_refresh_token_from_token_data
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.domain import Token as TokenDomain
from passport.backend.social.common.useragent import Url
from passport.backend.social.proxylib import (
    get_proxy,
    useragent,
)
from passport.backend.social.proxylib.proxy import parse_access_token_from_authorization_code_response
from passport.backend.utils.string import smart_text
from werkzeug.urls import url_decode


_YANDEX_REDIRECTOR_DOMAIN_GROUP = DomainGroup(
    [
        set(['ru', 'com', 'ua', 'by', 'kz', 'net']),
        set(['yandex']),
        set(['social', 'social-test', 'social-rc']),
    ],
    [
        set(['tr']),
        set(['com']),
        set(['yandex']),
        set(['social', 'social-test', 'social-rc']),
    ],
    [
        set(['ru', 'com', 'ua', 'by', 'kz', 'net']),
        set(['yandex']),
        set(['social-dev']),
        set([str(i) for i in range(10)])
    ],
    [
        set(['tr']),
        set(['com']),
        set(['yandex']),
        set(['social-dev']),
        set([str(i) for i in range(10)])
    ],
)

# Эти приложения передают в ручки Социализма имя от чужого приложения,
# а токен от своего.
_APPLICATIONS_ALLOWED_TO_FIX_CLIENT_ID = frozenset([
    'vkontakte-afisha-mobile',
    'vkontakte-drive',
    'vkontakte-market-mobile',
    'vkontakte-metrica-mobile',
    'vkontakte-money-mobile',
    'vkontakte-music',
    'vkontakte-radio',
    'vkontakte-search-mobile',
])


logger = logging.getLogger('social.broker.communicators')


def get_communicator_application_mapper():
    return LazyLoader.get_instance('communicator_application_mapper')


class Communicator(object):
    # Получение authorize url происходит без дополнительных запросов к провайдеру
    INSTANT_AUTHORIZE_URL = True

    OAUTH_REQUEST_TOKEN_URL = None
    OAUTH_AUTHENTICATE_URL = None
    REDIRECT_NEEDED = False

    # По сути является не дефолтным набором скоупов, а минимальным (смотри
    # метод Communicator.get_scope)
    # TODO: Подумать о переименовании в minimal_scopes
    default_scopes = []

    scope_delimiter = ','
    scope_name = 'scope'
    IS_CALLBACK_IN_STATE = False

    # Параметр state из authorization_url будет случайной строкой. По этой
    # строке можно будет найти callback_url.
    IS_OPAQUE_STATE = False

    METHODS = None
    PROFILE_FIELDS = None

    display_value_default = 'popup'
    display_value_if_not_set = 'popup'
    display_available = ['page', 'popup', 'touch', 'wap']
    display_parameter_name = 'display'

    display_map = dict([(x, display_value_default) for x in display_available])

    # Коды языков в форме ISO 639-1
    SUPPORTED_UI_LANGUAGES = {'ru'}
    DEFAULT_UI_LANGUAGE = 'ru'

    def __init__(self, app, display=None, ui_language=None, experiments=None):
        self.retries = self._get_app_setting(app, 'retries') or 2
        self.timeout = self._get_app_setting(app, 'timeout') or 5

        if app.provider:
            self.provider_code = app.provider['code']
        else:
            self.provider_code = None

        self.app = app

        if not display:
            self.display = self.display_value_default
        elif display not in self.display_map:
            raise DisplayInvalidError('Display value "%s" is not valid. Valid values: %s' % (display, self.display_map.keys()))
        else:
            self.display = self.display_map[display]

        self._ui_language = ui_language

        self._experiments = experiments or list()

    def _get_app_setting(self, app, name):
        if app.provider:
            provider_value = app.provider.get(name)
        else:
            provider_value = None
        global_value = getattr(social_config, name, None)
        return provider_value or global_value

    @staticmethod
    def create(app, display=None, ui_language=None, experiments=None):
        if app.provider:
            communicator_application_mapper = get_communicator_application_mapper()
            communicator_cls = communicator_application_mapper[(app.provider['code'], app)]
            communicator = communicator_cls(
                app=app,
                display=display,
                ui_language=ui_language,
                experiments=experiments,
            )
        else:
            communicator = ApplicationOnlyOAuth2Communicator(
                app,
                display,
                ui_language,
                experiments,
            )
        return communicator

    def get_scope(self, scopes=None):
        """
        Получить значение параметра scope для запроса к провайдеру.
        Если существует обязательный scope, то он объединяется с запрашиваемым.
        Args:
         scopes - list of strings, права доступа для провайдера сверх дефолтных
        Returns:
         string - scope string for provider OR None when no scopes.
        """
        scope_result = self.default_scopes
        if scopes:
            if self.app.allowed_scope:
                scopes = list(set(scopes) & set(split_scope_string(self.app.allowed_scope)))
            scope_result = list(set(scope_result + scopes))

        if scope_result:
            return self.scope_delimiter.join(sorted(scope_result))

    def get_real_token_data(self, token, fail_silent=False, need_client_id=True):
        return token

    def check_token(self, token):
        raise NotImplementedError()  # pragma: no cover

    def sanitize_server_token(self, server_token, refresh_token):
        token_dict = self._collect_info_about_token(
            server_token,
            refresh_token,
            fail_silent=True,
            need_client_id=False,
        )
        self._update_token_from_token_dict(token_dict, server_token, refresh_token)
        return token_dict

    def sanitize_client_token(self, client_token):
        server_token, refresh_token = self._client_token_to_server_token(client_token)
        token_dict = self._collect_info_about_token(
            server_token,
            refresh_token,
            fail_silent=False,
            need_client_id=True,
        )
        self.check_token(token_dict)
        self._update_token_from_token_dict(token_dict, server_token, refresh_token)
        return server_token, refresh_token

    def build_nonce(self):
        return

    def check_nonce(self, token_dict, nonce):
        return

    def client_token_to_social_userinfo(self, client_token):
        """
        Иногда какие-то данные о профиле провайдер предоставляет только
        в/по/вместе с клиентским токеном. В таких случаях этот метод поможет
        извлечь доступные данные.
        """
        return dict()

    def _build_redirect_args(
        self,
        app,
        callback_url,
    ):
        """
        Строит словарь с параметрами, которые описывают социальному провайдеру
        адрес, на который нужно вернуть результат сессии с пользователем.
        """
        if self.IS_CALLBACK_IN_STATE:
            return {
                'redirect_uri': social_config.domain_to_redirect_url[app.domain],
                'state': callback_url,
            }
        if self.IS_OPAQUE_STATE:
            return {
                'redirect_uri': social_config.domain_to_redirect_url[app.domain],
                'state': smart_text(self._get_state_from_callback_url(callback_url)),
            }

        callback_domain = furl(callback_url).host
        is_redirect_required = self.REDIRECT_NEEDED or not is_subdomain(callback_domain, app.domain)

        if is_redirect_required:
            redirect_uri = self._build_redirect_uri(app.domain, callback_url)
            query_string = urlencode({'url': callback_url})
            uri = '%(redirect_uri)s?%(query)s' % {
                'redirect_uri': redirect_uri,
                'query': query_string,
            }
        else:
            uri = callback_url
        return {'redirect_uri': uri}

    @classmethod
    def _build_redirect_uri(cls, domain, callback_url):
        """
        Строит адрес для перенаправления пользователя после сеанса связи с
        социальным провайдером.
        """
        # Собираем хост в конечном редиректе: инстанс-часть берём из
        # приложения, если она там задана, а если нет - то используем
        # текущий инстанс; плюс яндекс-домен из приложения.
        logger.debug('Processing callback url...')
        original_callback_url = callback_url
        redirector_uri = social_config.domain_to_redirect_url[domain]

        if is_domain_mask(domain):
            callback_domain = furl(callback_url).host
            if callback_domain in _YANDEX_REDIRECTOR_DOMAIN_GROUP:
                hybrid_domain = _YANDEX_REDIRECTOR_DOMAIN_GROUP.find_hybrid(
                    start=callback_domain,
                    end=domain[1:],
                )

                if hybrid_domain:
                    redirector_uri = furl(redirector_uri)
                    redirector_uri.host = hybrid_domain
                    redirector_uri = str(redirector_uri)
        logger.debug('Initial url: "%s", resulting url: "%s"' % (
            original_callback_url,
            redirector_uri,
        ))
        return redirector_uri

    def _get_real_token_data(self, token, fail_silent=False, need_client_id=True):
        logger.debug('Requesting access_token info...')

        proxy = get_proxy(self.provider_code, token, self.app, should_retry_on_invalid_token=True)
        token_info = {}

        try:
            token_info = proxy.get_token_info(need_client_id=need_client_id)
        except (
            InvalidTokenProxylibError,
            SocialUserDisabledProxylibError,
        ):
            raise

        except NetworkProxylibError as e:
            if fail_silent:
                logger.warning('Failed to get access_token scopes because of network error')
                ErrorHandler(e, handler=None).exception_to_graphite()
            else:
                raise

        except ProviderRateLimitExceededProxylibError as e:
            if fail_silent:
                logger.warning('Failed to get access_token scopes because of rate limit exceeded')
                ErrorHandler(e, handler=None).exception_to_graphite()
            else:
                raise

        except BasicProxylibError as e:
            if fail_silent:
                logger.error('Failed to get access_token scopes. %s: %s', type(e).__name__, str(e))
                ErrorHandler(e, handler=None).exception_to_graphite()
            else:
                raise

        logger.debug('Token info: %s' % token_info)

        if 'scopes' in token_info:
            token['scope'] = ','.join(token_info['scopes'])

        if 'client_id' in token_info:
            token['client_id'] = token_info['client_id']

        if 'expires' in token_info:
            token['expires'] = token_info['expires']

        return token

    def _check_token(self, token):
        if self.app.id != token['client_id']:
            real_app = providers.get_application_by_provider_app_id(self.app.provider['id'], token['client_id'])
            if not real_app:
                raise OAuthTokenInvalidError('Unknown client_id: %s' % token['client_id'])
        else:
            real_app = self.app

        if token['application'] != real_app.identifier:
            wrong_app = providers.get_application_by_id(token['application'])
            if real_app.name not in _APPLICATIONS_ALLOWED_TO_FIX_CLIENT_ID:
                raise OAuthTokenInvalidError('Token is not valid for application %s' % wrong_app.name)
            logger.debug('Fix wrong client_id %s to %s' % (wrong_app.name, real_app.name))
            token['application'] = real_app.identifier

    def _collect_info_about_token(self, server_token, refresh_token, fail_silent=False,
                                  need_client_id=True):
        token_dict = server_token.to_dict_for_proxy()
        if refresh_token:
            refresh_token.update_token_dict_for_proxy(token_dict)

        token_dict = self.get_real_token_data(token_dict, fail_silent, need_client_id)
        return token_dict

    def _update_token_from_token_dict(self, token_dict, server_token, refresh_token):
        server_token.update_from_dict_for_proxy(token_dict)
        if refresh_token:
            refresh_token.update_from_token_dict_for_proxy(token_dict)

    def _client_token_to_server_token(self, client_token):
        logger.debug('Exchanging access_token')

        token_dict = client_token.to_dict_for_proxy()
        proxy = get_proxy(self.provider_code, token_dict, self.app)
        if not hasattr(proxy, 'exchange_token'):
            logger.debug('Token can not be exchanged for this provider')
            return client_token, None

        token_dict = dict(token_dict)
        token_dict.update(proxy.exchange_token())

        server_token = TokenDomain.from_dict_for_proxy(token_dict)
        refresh_token = build_refresh_token_from_token_data(token_dict)

        return server_token, refresh_token

    def _on_parse_access_token_parsing_error(self, response):
        logger.warning('Failed to parse access token response: %s' % trim_message(response))
        raise CommunicationFailedError(
            'Failed to parse access token response',
            failure_source_type=FailureSourceType.external,
        )

    def _on_parse_access_token_invalid_grant_error(self, response):
        logger.warning('Failed to exchange authorization code to access_token: %s' % trim_message(response))
        raise CommunicationFailedError(
            'Failed to exchange authorization code to access_token',
            failure_source_type=FailureSourceType.not_error,
        )

    def _get_ui_language(self):
        if self._ui_language in self.SUPPORTED_UI_LANGUAGES:
            lang = self._ui_language
        else:
            lang = self.DEFAULT_UI_LANGUAGE
        return lang

    def _get_service_for_useragent(self):
        if self.provider_code:
            return self.provider_code
        if self.app:
            return self.app.name

    def _get_state_from_callback_url(self, callback_url):
        return Oauth2State.find_by_url(callback_url)

    def get_task(self):
        return request_ctx.task


class OAuthCommunicator(Communicator):
    """
    Abstract implementation of OAuth 1.0
    """
    OAUTH_REQUEST_TOKEN_URL = None
    OAUTH_ACCESS_TOKEN_URL = None
    OAUTH_AUTHORIZE_URL = None
    REQUEST_METHOD = 'GET'

    INSTANT_AUTHORIZE_URL = False

    REQUEST_SIGNER_CLASS = Oauth1RequestSigner

    def get_authorize_url(self, options):
        logger.debug('[communicator] Getting OAuth redirect url...')
        redirect_args = self._build_redirect_args(self.app, options.callback_url)
        url, body, headers = self.get_request_token_url(
            method=self.REQUEST_METHOD,
            scope=options.scope,
        )
        logger.debug('url="%s"' % url)
        response = useragent.execute_request(
            method=self.REQUEST_METHOD,
            url=url,
            fields=body,
            headers=headers,
            retries=self.retries,
            timeout=self.timeout,
            request_signer=self.get_request_signer(oauth_callback=redirect_args['redirect_uri']),
            from_intranet=self.app.request_from_intranet_allowed,
            service=self._get_service_for_useragent(),
        )

        logger.debug('Getting request token from response...')
        request_token = self.parse_request_token(response.decoded_data)
        args = dict(oauth_token=request_token.value)

        url = furl(self.OAUTH_AUTHENTICATE_URL).add(args).url
        return url, request_token

    def get_request_token_url(self, method='POST', scope=None, data=None):
        method = method.upper()

        headers = dict()
        url = furl(self.OAUTH_REQUEST_TOKEN_URL)
        body = dict() if not data else dict(data)

        if scope:
            if method == 'GET':
                url.query.params[self.scope_name] = scope
            elif method == 'POST':
                body[self.scope_name] = scope

        return url.url, body, headers

    def get_request_signer(self, oauth_callback=None, oauth_token=None,
                           oauth_token_secret=None, verifier=None):
        return self.REQUEST_SIGNER_CLASS(
            client_id=self.app.id,
            client_secret=self.app.secret,
            oauth_callback=oauth_callback,
            oauth_token=oauth_token,
            oauth_token_secret=oauth_token_secret,
            verifier=verifier,
        )

    def get_access_token_request(self):
        return self.OAUTH_ACCESS_TOKEN_URL, None, None

    def parse_access_token(self, response):
        raw_response = response
        response = self._access_token_response_str_to_dict(raw_response)
        self._check_error_in_access_token_response(response, raw_response)
        try:
            token = {
                'value': response['oauth_token'],
                'secret': response['oauth_token_secret'],
                'expires': expires_in_to_expires_at(response.get('oauth_expires_in')),
            }
        except KeyError:
            self._on_parse_access_token_parsing_error(raw_response)
        return token

    def _access_token_response_str_to_dict(self, response):
        raw_response = response
        response = url_decode(raw_response.encode('utf-8'))
        keys = response.keys()
        if len(keys) == 0 or (len(keys) == 1 and not response[keys[0]]):
            self._on_parse_access_token_parsing_error(raw_response)
        return response

    def _check_error_in_access_token_response(self, response, raw_response):
        if 'error' in response:
            self._on_parse_access_token_parsing_error(raw_response)

        problem = response.get('oauth_problem')
        # TODO
        # Пока возвращается token_not_renewable, ЖЖшники обещали поправить
        # на что-то другое.
        if problem in ['token_not_renewable', 'permission_denied']:
            raise UserDeniedError()
        elif problem:
            error_msg = 'Unexpected error in access token response: %s' % problem
            logger.error(error_msg)
            raise CommunicationFailedError(error_msg)

    def parse_request_token(self, response):
        """
        Типичный ответ:
        oauth_token=snb4ybB6VKyZ717wuG3A4omjl9YgrS6asaqLztmEs&oauth_token_secret
        =2GtxUEnqR5oZfKtSvhmZVHexOuFCpTJ1ofuBCdlU&oauth_callback_confirmed=true
        """
        logger.debug('On request token response ' + trim_message(response))

        query = url_decode(response.strip().encode('utf-8'))

        try:
            confirmed = query['oauth_callback_confirmed']
            if confirmed != 'true':
                raise CommunicationFailedError('Request token is not confirmed')

            token = TokenModel()
            token.value = query['oauth_token']
            token.secret = query['oauth_token_secret']
            token.application = self.app
        except KeyError as e:
            msg = ('Can not extract request token from %s because of %s' % (response, e))
            logger.error('%s' % msg, exc_info=True)
            raise CommunicationFailedError(msg, failure_source_type=FailureSourceType.external)
        return token

    def has_error_in_callback(self, query, request_token_value=None):
        if 'denied' in query:
            if request_token_value and query['denied'] != request_token_value:
                raise CommunicationFailedError('Callback with incorrect "oauth_token" and access denial')
            raise UserDeniedError('User denied access')
        oauth_token = query.get('oauth_token', None)
        if not oauth_token:
            raise CommunicationFailedError('No "oauth_token" in callback found')
        # проверка на oauth_token, должен совпадать с тем, что получен
        # на предыдущем этапе (obtain request token)
        if request_token_value and oauth_token != request_token_value:
            raise CommunicationFailedError('Callback with incorrect "oauth_token"')

    def get_exchange_value_from_callback(self, query):
        logger.debug('Exchange value: %s' % query)
        if 'oauth_verifier' not in query:
            raise CommunicationFailedError('No exchange value specified')
        return query['oauth_verifier']

    def get_access_token(
        self,
        exchange,
        callback_url,
        scopes,
        request_token,
    ):
        url, data, headers = self.get_access_token_request()
        request_signer = self.get_request_signer(
            oauth_token=request_token['value'],
            oauth_token_secret=request_token['secret'],
            verifier=exchange,
        )

        response = useragent.execute_request(
            method='POST' if data else 'GET',
            url=url,
            fields=data,
            headers=headers,
            retries=self.retries,
            timeout=self.timeout,
            request_signer=request_signer,
            from_intranet=self.app.request_from_intranet_allowed,
            service=self._get_service_for_useragent(),
        )

        token = self.parse_access_token(response.decoded_data)
        token['scope'] = self.get_scope(scopes)
        return token


class OAuth2Communicator(Communicator):
    """Abstract implementation of OAuth v2"""

    OAUTH_ACCESS_TOKEN_URL = None
    OAUTH_AUTH_TYPE_BASIC = False
    ACCESS_TOKEN_REQUEST_TYPE = 'GET'
    ACCESS_TOKEN_RESPONSE_FORMAT = 'json'

    OAUTH_AUTHORIZE_URL = None
    OAUTH_AUTHORIZE_MANDATORY_ARGUMENTS = {}

    # Этот параметр нужно передать в ручку authorize, для того чтобы соц.
    # провайдер обязательно спросил разрешение у пользователя (а Google выдал
    # нам refresh_token).
    _force_prompt_args = {}

    def get_authorize_url(self, options):
        logger.debug('Getting authorize url...')
        query = self._get_authorize_query(options)

        # Костыль для авторизации из приложений Алисы через яндексовый Oauth
        url = Url(self.OAUTH_AUTHORIZE_URL)
        if url.paramless == social_config.yandex_oauth_authorize_url:
            query.update({'force_confirm': 'yes'})

        return str(url.add_params(query))

    def _get_authorize_query(self, options):
        query = {}
        query.update(self.OAUTH_AUTHORIZE_MANDATORY_ARGUMENTS)
        query.update({
            'client_id': options.client_id or self.app.custom_provider_client_id or self.app.id,
            'response_type': 'code',
        })

        if self.display != self.display_value_if_not_set and self.display_parameter_name:
            query[self.display_parameter_name] = self.display

        if options.scope:
            query[self.scope_name] = options.scope

        if options.force_prompt:
            query.update(self._force_prompt_args)

        if options.login_hint:
            query.update(self._build_login_hint_args(options.login_hint))

        if options.yandex_auth_code is not None:
            query['yandex_auth_code'] = options.yandex_auth_code

        if options.nonce:
            query['nonce'] = options.nonce

        if options.user_param and self.app.authorize_user_param:
            query[self.app.authorize_user_param.name] = options.user_param

        query.update(self._build_redirect_args(self.app, options.callback_url))

        return query

    def get_access_token_request(
        self,
        callback_url=None,
        client_id=None,
        client_secret=None,
        code=None,
        scopes=None,
    ):
        logger.debug('[communicator] Getting OAuth2 access token request...')

        if self.OAUTH_AUTH_TYPE_BASIC:
            get_request = oauth2.token.build_authorization_code_request_basic_auth
        else:
            get_request = oauth2.token.build_authorization_code_request

        request = get_request(
            endpoint=self.OAUTH_ACCESS_TOKEN_URL,
            code=code,
            client_id=client_id or self.app.custom_provider_client_id or self.app.id,
            client_secret=client_secret or self.app.secret or '',
            http_method=self.ACCESS_TOKEN_REQUEST_TYPE,
        )

        redirect_args = self._build_redirect_args(self.app, callback_url)
        redirect_args.pop('state', None)
        if request.data:
            request.data.update(redirect_args)
        else:
            request.query.update(redirect_args)

        if self.app.token_user_param and self.get_task().user_param:
            if self.app.token_user_param.type == UserParamDescriptor.TYPE_HEADER:
                params = request.headers
            else:
                params = request.data
            params[self.app.token_user_param.name] = self.get_task().user_param

        full_url = str(Url(request.endpoint).add_params(request.query))

        return full_url, request.data, request.headers

    def get_exchange_value_from_callback(self, query):
        if 'code' not in query:
            raise MissingExchangeValueCommunicationFailedError(
                'No exchange value specified',
                query,
            )
        return query['code']

    def has_error_in_callback(self, query, *args, **kwargs):
        error_name = query.get('error')
        if error_name:
            if error_name == 'access_denied':
                raise UserDeniedError('User denied access', query)
            raise AuthorizationErrorResponseCommunicationFailedError(
                'Error in callback: ' + query['error'],
                query,
            )

    def _build_login_hint_args(self, login_hint):
        return dict()

    def _get_error_detector_for_access_token_response(self):
        return

    def get_access_token(
        self,
        exchange,
        callback_url,
        scopes,
        request_token,
    ):
        url, data, headers = self.get_access_token_request(
            callback_url=callback_url,
            code=exchange,
            scopes=scopes,
        )

        try:
            response = useragent.execute_request(
                method='POST' if data else 'GET',
                url=url,
                data=data,
                headers=headers,
                retries=self.retries,
                timeout=self.timeout,
                from_intranet=self.app.request_from_intranet_allowed,
                parser=self._parse_access_token_response,
                service_temporary_unavailable_exceptions=[
                    CommunicationFailedError,
                ],
                service=self._get_service_for_useragent(),
            )
        except ProviderTemporaryUnavailableProxylibError as e:
            service_temporary_unavailable_exception = copy(
                e.service_temporary_unavailable_exception,
                fallback=e.service_temporary_unavailable_exception,
            )
            service_temporary_unavailable_exception.cause = e
            raise service_temporary_unavailable_exception

        token = response.parsed
        token['scope'] = self.get_scope(scopes)
        return token

    def _parse_access_token_response(self, http_response):
        if http_response.status // 100 == 5:
            raise BadHttpStatusCommunicationFailedError(http_response.status)
        return self.parse_access_token(http_response.decoded_data)

    def parse_access_token(self, response):
        try:
            return parse_access_token_from_authorization_code_response(
                response,
                self.ACCESS_TOKEN_RESPONSE_FORMAT,
                self._get_error_detector_for_access_token_response(),
            )
        except (
            ProviderTemporaryUnavailableProxylibError,
            UnexpectedResponseProxylibError,
        ):
            self._on_parse_access_token_parsing_error(response)
        except InvalidTokenProxylibError:
            self._on_parse_access_token_invalid_grant_error(response)


class OAuth2CommunicatorWithAbsoluteRedirect(OAuth2Communicator):
    IS_CALLBACK_IN_STATE = True


class ApplicationOnlyOAuth2Communicator(OAuth2CommunicatorWithAbsoluteRedirect):
    ACCESS_TOKEN_REQUEST_TYPE = 'POST'
    REDIRECT_NEEDED = True
    scope_delimiter = ' '

    def __init__(self, app, display, ui_language=None, experiments=None):
        super(ApplicationOnlyOAuth2Communicator, self).__init__(
            app,
            display,
            ui_language,
            experiments,
        )

        assert app.authorization_url
        assert app.token_url
        self.OAUTH_AUTHORIZE_URL = app.authorization_url
        self.OAUTH_ACCESS_TOKEN_URL = app.token_url
        self.default_scopes = split_scope_string(app.default_scope)

    def _get_error_detector_for_access_token_response(self):
        return self._error_detector_for_access_token_response

    @staticmethod
    def _error_detector_for_access_token_response(response):
        try:
            return oauth2.token.detect_error(response)
        except oauth2.token.InvalidClient:
            raise InvalidTokenProxylibError()


class CommunicatorApplicationMapper(ApplicationMapper):
    def __init__(self):
        super(CommunicatorApplicationMapper, self).__init__()

        for module in self._discover_communicator_modules():
            custom_add_to_manager = getattr(module, 'add_to_manager', None)
            if custom_add_to_manager:
                custom_add_to_manager(self)
            else:
                communicators = self._discover_communicators(module)
                for communicator in communicators:
                    self.add_mapping(communicator.provider_code, communicator)

    @classmethod
    def _discover_communicators(cls, module):
        module_attrs = [getattr(module, n) for n in dir(module)]
        communicators = []
        for attr in module_attrs:
            try:
                if issubclass(attr, Communicator):
                    communicator_cls = attr
                    if getattr(communicator_cls, 'provider_code', None):
                        communicators.append(communicator_cls)
            except TypeError:
                pass
        return communicators

    @classmethod
    def _discover_communicator_modules(cls):
        import passport.backend.social.broker.communicators
        modules = []
        for _, name, is_pkg in walk_packages(passport.backend.social.broker.communicators.__path__):
            assert not is_pkg
            modules.append(
                import_module(
                    '.' + name,
                    passport.backend.social.broker.communicators.__name__,
                ),
            )
        return modules


class Oauth2State(object):
    def __init__(self):
        self._state = None

    @classmethod
    def find_by_url(cls, url):
        # Callback Url выглядит так https://social.yandex.ru/broker2/12345/callback
        bits = list(Url(url).path.split('/'))
        task_id = bits[bits.index('callback') - 1]
        state = str(uuid.UUID(task_id))

        self = cls()
        self._state = state
        return self

    def __str__(self):
        return self._state or ''


class AuthorizeOptions(object):
    def __init__(
        self,
        callback_url,
        client_id=None,
        scope=None,
        force_prompt=False,
        yandex_auth_code=None,
        login_hint=None,
        nonce=None,
        user_param=None,
    ):
        self.callback_url = callback_url
        self.client_id = client_id
        self.force_prompt = force_prompt
        self.login_hint = login_hint
        self.nonce = nonce
        self.scope = scope
        self.user_param = user_param
        self.yandex_auth_code = yandex_auth_code
