# -*- coding: utf-8 -*-

from __future__ import unicode_literals


def is_subdomain(subdomain, domain):
    if is_domain_mask(domain):
        return subdomain.endswith(domain)
    else:
        return subdomain == domain


def is_domain_mask(domain):
    return domain[0] == '.'


class DomainGroup(object):
    """
    Доменная группа
    """
    def __init__(self, *domain_groups):
        """
        Входные параметры:
            domain_groups
                Дескрипторы доменной группы.

        Дескриптор доменной группы -- это список из множеств.

        Каждое множество соответсвует уровню доменного имени. В множестве
        находятся имена допустимые на этом уровне.

        Например, дескриптор
            [
                {'ru', 'net'},
                {'yandex'},
                {'foo', 'bar'},
            ]
        описывает только доменные имена foo.yandex.ru, foo.yandex.net,
        bar.yandex.ru и bar.yandex.net.

        Если какие-то имена из последних уровней можно опустить в домене, то в
        соответствующем этому уровню множесте нужно использовать None.

        Например, дескриптор
            [
                {'ru', 'net'},
                {'yandex', None},
            ]
        описывает только доменные именя yandex.ru, yandex.net, ru, net.
        """
        self._domain_groups = domain_groups

    def find_hybrid(self, start, end):
        """
        Найти в группе доменов имя = start[:n] + end, при этом n такое что
        start[:n+1] + end не входит в группу.

        Входные параметры
            start, end
                Доменные имена

        Выходные параметры
            Гибридное доменное имя, если такое нашлось в группе.
            Иначе None.
        """
        start_bits = start.split('.')
        end_bits = end.split('.')

        hybrid = None
        while start_bits:
            hybrid = '.'.join(start_bits + end_bits)
            if hybrid in self:
                break
            else:
                hybrid = None
            start_bits.pop()
        return hybrid

    def __contains__(self, domain):
        domain_bits = domain.split('.')
        domain_bits.reverse()
        return any(
            self._domain_group_has_domain(domain_group, domain_bits)
            for domain_group in self._domain_groups,
        )

    @classmethod
    def _domain_group_has_domain(cls, domain_group, domain_bits):
        domain_group = list(domain_group) + [set([None])]
        for domain_bit in domain_bits:
            names = domain_group.pop(0)
            if domain_bit not in names:
                return False
        names = domain_group.pop(0)
        return None in names
