# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.broker.exceptions import (
    AuthorizationRequiredError,
    CommunicationFailedError,
    DatabaseFailedError,
    GrantsMissingError,
    InvalidParametersError,
    OAuthTokenInvalidError,
    RateLimitExceededError,
    SessionInvalidError,
    SocialBrokerError,
    UserDeniedError,
    UserRejectedBindError,
)
from passport.backend.social.broker.misc import generate_retpath
import passport.backend.social.common.exception
from passport.backend.social.common.exception import FailureSourceType
from passport.backend.social.proxylib.error_handler import ErrorHandler as _ErrorHandler


logger = logging.getLogger(__name__)


def process_error(exception, handler):
    """
    По типу брошенного исключения выбираем что вернуть пользователю, а именно:
        - Непонятное исключение — 500 с текстом исключения и error_code = 1
        - остальное - 400, error_code исключения и сообщением сключения.
    Returns:
        Объект Response с заполненным http-кодом, телом и mimetype.
    """
    error_handler = ErrorHandler(exception, handler)
    response = error_handler.exception_to_response()
    error_handler.exception_to_graphite()
    return response


class ErrorHandler(_ErrorHandler):
    def __init__(self, exception, handler):
        super(ErrorHandler, self).__init__(exception)
        self._handler = handler

    def exception_to_response(self):
        if self._exception_to_error_type() == FailureSourceType.internal:
            logger.error('Internal error', exc_info=True)
        elif isinstance(self._exception, passport.backend.social.common.exception.CaptchaNeededProxylibError):
            logger.error('Request failed because of CAPTCHA required')
        elif isinstance(self._exception, passport.backend.social.common.exception.ValidationRequiredProxylibError):
            logger.error('Request failed because of validation required')
        elif isinstance(self._exception, passport.backend.social.common.exception.ProviderRateLimitExceededProxylibError):
            logger.warning('Request failed because of rate limit exceeded')
        return _build_error_response(self._exception, self._handler)

    def _exception_to_error_type(self):
        error_type = _exception_to_error_type(self._exception)
        if error_type != FailureSourceType.internal:
            return error_type
        return super(ErrorHandler, self)._exception_to_error_type()


def _build_error_response(exception, handler):
    error_code, error_message = _exception_to_error_code(exception)
    response = {
        'error': {
            'code': error_code,
            'message': error_message,
        },
        'request_id': handler.request.id,
    }

    args = handler.processed_args

    if args.get('provider'):
        response['provider'] = args['provider']

    response.update(_get_args_for_retry(handler))

    if args.get('retpath') is not None:
        additional_args = {}
        if handler.should_frontend_passthrough_error(exception):
            additional_args.update({
                'code': error_code,
                'request_id': handler.request.id,
            })
            response['passthrough'] = True
        response['retpath'] = generate_retpath(
            args['retpath'],
            args.get('place'),
            status='error',
            additional_args=additional_args,
        )

    http_status_code = _exception_to_http_status_code(exception)

    handler.response.data = handler.compose_json_response(response)
    handler.response.status_code = http_status_code
    handler.response.api_status = 'error'
    handler.response.api_error_code = error_code
    return handler.response


def _get_args_for_retry(handler):
    if 'retry_url' not in handler.processed_args:
        return dict()
    if not handler.task_can_be_retried():
        return dict()
    cookie = handler.save_task_to_cookie()
    return dict(
        retry_url=handler.processed_args['retry_url'],
        cookies=[cookie],
    )


def _exception_to_error_code(exception):
    if isinstance(exception, SocialBrokerError):
        error_code = exception.error_code
        message = unicode(exception)
    elif isinstance(exception, passport.backend.social.common.exception.ProviderRateLimitExceededProxylibError):
        error_code = RateLimitExceededError.__name__
        message = exception.args[0] if exception.args else 'Rate limit exceeded'
    elif isinstance(
        exception,
        (
            passport.backend.social.common.exception.ProviderCommunicationProxylibError,
            passport.backend.social.common.exception.SocialUserDisabledProxylibError,
            passport.backend.social.common.exception.PermissionProxylibError,
            passport.backend.social.common.exception.CaptchaNeededProxylibError,
            passport.backend.social.common.exception.ValidationRequiredProxylibError,
        ),
    ):
        error_code = CommunicationFailedError.__name__
        message = exception.message or 'Error while communicating with social provider'
    elif isinstance(exception, passport.backend.social.common.exception.InvalidTokenProxylibError):
        error_code = OAuthTokenInvalidError.__name__
        message = exception.message or 'Invalid token'
    elif isinstance(exception, passport.backend.social.common.exception.ApplicationUnknown):
        error_code = 'ApplicationUnknownError'
        message = 'Given application or application in given task not found'
    elif isinstance(exception, passport.backend.social.common.exception.DatabaseError):
        error_code = DatabaseFailedError.error_code
        message = ''
    else:
        error_code = 'InternalError'
        message = unicode(exception)
    return error_code, message


def _exception_to_http_status_code(exception):
    if isinstance(exception, passport.backend.social.common.exception.ProviderTemporaryUnavailableProxylibError):
        response_code = 503
    elif isinstance(exception, passport.backend.social.common.exception.ApplicationUnknown):
        response_code = 400
    elif isinstance(
        exception,
        (
            passport.backend.social.common.exception.ProviderRateLimitExceededProxylibError,
            passport.backend.social.common.exception.ProviderCommunicationProxylibError,
            passport.backend.social.common.exception.InvalidTokenProxylibError,
        ),
    ):
        response_code = 400
    elif isinstance(
        exception,
        (
            InvalidParametersError,
            SessionInvalidError,
            AuthorizationRequiredError,
            UserDeniedError,
            UserRejectedBindError,
            RateLimitExceededError,
            OAuthTokenInvalidError,
            CommunicationFailedError,
            GrantsMissingError,
        ),
    ):
        response_code = 400
    else:
        response_code = 500
    return response_code


def _exception_to_error_type(exception):
    if isinstance(exception, passport.backend.social.common.exception.ApplicationUnknown):
        error_type = FailureSourceType.not_error
    else:
        error_type = FailureSourceType.internal
    return error_type
