# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.social.common.exception import (
    CauseExceptionMixin,
    DatabaseFailureSourceTypeMixin,
    NotErrorFailureSourceTypeMixin,
)
from passport.backend.social.common.web_service import description_for_consumer_from_grants_missing_error


class SocialBrokerError(CauseExceptionMixin, Exception):
    """Базовый класс для всех наших исключений, не надо инстанцировать его напрямую"""
    error_code = None

    def __init__(self, *args, **kwargs):
        super(SocialBrokerError, self).__init__(*args)

        self._kwargs = kwargs
        for key, val in kwargs.iteritems():
            setattr(self, key, val)

        self.setup_cause()

    def __copy__(self):
        cls = type(self)
        return cls(*self.args, **self._kwargs)


class InvalidSessionidError(SocialBrokerError):
    """
    Исключение нужно для внутренних нужд. Кажется, никогда наружу его пробрасывать не будем
    """
    error_code = 'InvalidSessionidError'


class UserDeniedError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Пользователь не дал приложению доступ"""
    error_code = 'UserDeniedError'

    def __init__(self, description='', query=None):
        super(UserDeniedError, self).__init__(description, query=query)


class UserRejectedBindError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Пользователь не дал согласия на привязку профиля к аккаунту"""
    error_code = 'UserRejectedBindError'


class SessionInvalidError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Проблемы взаимодействия с клиентом: не тот порядок запросов, запрос по несуществующему task_id и т. д."""
    error_code = 'SessionInvalidError'


class CommunicationFailedError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Любые проблемы при взаимодействии с провайдерами: 404, ошибка в ответе (неверный code, любые другие ошибки)"""
    error_code = 'CommunicationFailedError'


class BadHttpStatusCommunicationFailedError(CommunicationFailedError):
    """
    Данное значение статуса в HTTP-ответе сервиса недопустимо
    """
    def __init__(self, http_status):
        super(BadHttpStatusCommunicationFailedError, self).__init__(http_status=http_status)


class AuthorizationErrorResponseCommunicationFailedError(CommunicationFailedError):
    """
    Получен отказ от ручки authorization
    """
    def __init__(self, description, query):
        super(AuthorizationErrorResponseCommunicationFailedError, self).__init__(description, query=query)


class MissingExchangeValueCommunicationFailedError(CommunicationFailedError):
    """
    Не найден authorization code в ответе ручки authorization
    """
    def __init__(self, description, query):
        super(MissingExchangeValueCommunicationFailedError, self).__init__(description, query=query)


class RateLimitExceededError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """При работе с соц провайдером был превышен лимит запросов."""
    error_code = 'RateLimitExceededError'


class OAuthTokenInvalidError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Переданный токен невалиден или не имеет всех необходимых прав"""
    error_code = 'OAuthTokenInvalidError'


class AuthorizationRequiredError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Выставлен флаг require_auth, но на ручке /start или /continue нет авторизации"""
    error_code = 'AuthorizationRequiredError'


class TaskNotFoundError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """
    Заданный таск не найден.
    """
    error_code = 'TaskNotFoundError'


class DatabaseFailedError(DatabaseFailureSourceTypeMixin, SocialBrokerError):
    """
    Сбой в работе одной из баз данных.
    """
    error_code = 'DatabaseFailedError'


class ProfileNotAllowedError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """
    Создание социального профиля запрещено.
    """
    error_code = 'ProfileNotAllowedError'


class GrantsMissingError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """
    Недостаточно грантов.
    """
    error_code = 'GrantsMissingError'

    def __str__(self):
        return description_for_consumer_from_grants_missing_error(self.error)


# --- InvalidParametersError

class InvalidParametersError(NotErrorFailureSourceTypeMixin, SocialBrokerError):
    """Один из переданных параметров неправильный"""
    error_code = 'InvalidParametersError'


class ConsumerUnknownError(InvalidParametersError):
    """Переданное имя потребителя не было найдено"""
    error_code = 'ConsumerUnknownError'


class ProviderUnknownError(InvalidParametersError):
    """Переданный код провайдера не найден"""
    error_code = 'ProviderUnknownError'


class ApplicationUnknownError(InvalidParametersError):
    """Попытка получить провайдера по несуществующему приложению"""
    error_code = 'ApplicationUnknownError'


class DisplayInvalidError(InvalidParametersError):
    """Попытка получить провайдера по несуществующему приложению"""
    error_code = 'DisplayInvalidError'


class RetpathInvalidError(InvalidParametersError):
    """Попытка получить провайдера по несуществующему приложению"""
    error_code = 'RetpathInvalidError'


class HostInvalidError(InvalidParametersError):
    """Не удалось получить известный TLD из переданного фронтендового домена"""
    error_code = 'HostInvalidError'


class UserIpInvalidError(InvalidParametersError):
    """Фронт не передал user_ip в форме"""
    error_code = 'UserIpInvalidError'


class AccessTokenError(InvalidParametersError):
    """Не пришел обязательный параметр provider_token, либо не пришел provider_token_secret для OAuth1.0 соц сети"""
    error_code = 'AccessTokenError'


class UseridInvalidError(InvalidParametersError):
    """Отсутствует обязательный параметр userid"""
    error_code = 'UseridInvalidError'


class PkceCodeInvalidError(InvalidParametersError):
    """
    Недопустимое значение Proof key of Code Exchange
    """
    error_code = 'PkceCodeInvalidError'


class PkceMethodInvalidError(InvalidParametersError):
    """
    Неизвестная хеш-функция для PKCE.
    """
    error_code = 'PkceMethodInvalidError'


class PkceVerifierInvalidError(InvalidParametersError):
    """
    Недопустимое значение открытого ключа в PKCE или не подходящее к значению
    закрытого ключа.
    """
    error_code = 'PkceVerifierInvalidError'


class TaskIdInvalidError(InvalidParametersError):
    """
    Недопустимое значение task_id.
    """
    error_code = 'TaskIdInvalidError'
