# -*- coding: utf-8 -*-

import heapq
import json
import logging

from passport.backend.social.common.misc import (
    dump_to_json_string,
    failsafe,
)
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.social_config import social_config


logger = logging.getLogger(__name__)


def build_failure_diagnostics_id(diagnostics_id):
    """
    Добавляем префикс, чтобы диагностики не смешались с другими объектами
    в Redis.
    """
    return ':fd:' + diagnostics_id


def load_failure_diagnostics(diagnostics_id):
    redis = get_redis()
    diagnostics_id = build_failure_diagnostics_id(diagnostics_id)
    diagnostics = redis.lrange(diagnostics_id, -social_config.diagnostics_per_report, -1)
    return [FailureDiagnostics.from_json(d) for d in diagnostics]


def save_failure_diagnostics(diagnostics_id, diagnostics_list, append=True):
    """
    Сохраняет список диагностик в Redis под именем diagnostics_id

    Входные параметры

        diagnostics_id -- идентификатор группы диагностик

        diagnostics_list -- список диагностик

        append -- признак, что данный список надо добавить к старому списку
            группы.

    Ограничения

    Максимальное число хранимых под одним именем диагностик ограничено. Когда
    число диагностик превышает максимальное, из группы удаляются самые старые
    диагностики. Таким образом в группе хранятся самые новые диагностики и их
    не больше MAX (как в кольцевом буфере).

    Группа диагностик хранится в БД ограниченное время В. Но каждый раз, когда
    в группу добавляются новые диагностики прошедшее время сбрасывается.
    """
    if len(diagnostics_list) > social_config.diagnostics_per_report:
        diagnostics_list = diagnostics_list[-social_config.diagnostics_per_report:]
    diagnostics_list = [d.to_json() for d in diagnostics_list]

    diagnostics_id = build_failure_diagnostics_id(diagnostics_id)
    redis = get_redis()

    if not append:
        redis.delete(diagnostics_id)
        if diagnostics_list:
            redis.rpush(diagnostics_id, *diagnostics_list)

    else:
        if diagnostics_list:
            length = redis.rpush(diagnostics_id, *diagnostics_list)
            if length > social_config.diagnostics_per_report:
                redis.ltrim(diagnostics_id, -social_config.diagnostics_per_report, -1)

    redis.expire(diagnostics_id, social_config.diagnostics_ttl)


class FailureDiagnostics(object):
    """
    Контейнер для хранения одной диагностики
    """
    def __init__(self, items):
        self._items = items

    def to_json(self):
        return dump_to_json_string(self._items, minimal=True)

    @classmethod
    def from_json(cls, doc):
        return cls(json.loads(doc))

    def to_response_dict(self):
        return self._items

    @property
    def context(self):
        """
        Контекст уточняет в каком сценарии была обнаружена проблема, когда
        одинаковая проблема появляется в разных сценария.
        """
        return self._items.get('context')

    @context.setter
    def context(self, value):
        self._items['context'] = value

    def __iter__(self):
        return self._items.iteritems()

    def __eq__(self, other):
        if not isinstance(other, FailureDiagnostics):
            return NotImplemented
        return dict(self) == dict(other)

    def __ne__(self, other):
        if not isinstance(other, FailureDiagnostics):
            return NotImplemented
        return not (self == other)

    def __repr__(self):
        cls = type(self)
        return '%s(%r)' % (cls.__name__, self._items)


class DiagnosticCouncil(object):
    """
    Базовый класс для описания экспертов, которые составят диагностику по
    данному исключению.

    Для данного исключения класс строится индекс для быстрого поиска по цепочке
    исключений, а затем составление диагностики делегируется методам-экспертам.

    Пример

    class DummyDiagnosticCouncil(DiagnosticCouncil):
        def get_experts(self):
            return [
                self.handle_unicode_error,
            ]

        def handle_unicode_error(self):
            error = self.find_first_exception((UnicodeDecodeError, UnicodeEncodeError))
            if error:
                return FailureDiagnostics(dict(type='unicode_error'))
    """

    def diagnose(self, exception, context=None):
        """
        Собирает диагностику о данном исключении
        """
        self.build_index(exception)
        return self._build_report(context)

    def find_exceptions(self, classinfo):
        """
        Ищет в цепочке исключения данного класса (или классов), затем
        возвращает их хронологическом порядке выбрасывания.

        Входные параметры

            classinfo -- класс исключения или кортеж
        """
        if not isinstance(classinfo, tuple):
            classinfo = (classinfo,)

        idx_lists = [self._table.get(cls, list()) for cls in classinfo]
        for idx in heapq.merge(*idx_lists):
            yield self._chain[idx]

    def find_first_exception(self, classinfo):
        for exc in self.find_exceptions(classinfo):
            return exc

    def get_experts(self):
        """
        Этот метод должен возвращать список функций-экспертов, которые
        составят диагностику по цепочке исключений хранимой в даннном объекте.
        """
        raise NotImplementedError()

    def build_index(self, exception):
        self._chain = self._traverse_exceptions_chain(exception)
        self._chain.reverse()

        self._table = dict()
        for i, exc in enumerate(self._chain):
            same_type_exceptions = self._table.setdefault(type(exc), list())
            same_type_exceptions.append(i)

    def _build_report(self, context=None):
        for expert in self.get_experts():
            report = expert()
            if report:
                if context is not None and report.context is None:
                    report.context = context
                return report

    def _traverse_exceptions_chain(self, exception):
        chain = list()
        seen = set()
        is_circular_list = False

        while not (exception is None or is_circular_list):
            chain.append(exception)
            seen.add(id(exception))
            exception = getattr(exception, 'cause', None)
            is_circular_list = id(exception) in seen

        if is_circular_list:
            # Такое может произойти, если не получилось скопировать исключение,
            # в таком случае в логах нужно найти сообщение об этом и научиться
            # копировать нужное исключение.
            logger.error('Diagnostic council detected a cycle in exceptions chain: %r' % chain)
        return chain


class DiagnosticManager(object):
    """
    Управляет сбором диагностики о исключениях или других необычных ситуациях,
    которые встретятся во время выполнения диагностируемой функции.

    Атрибуты объекта

        context -- позволяет указать контекст, чтобы можно было различать общую
            проблему происходящую в разных контекстах.

        is_fail -- признак того, что диагностируемая функция выбросила
            исключение или накопленную за время выполнения диагностику не
            получилось сохранить в БД.
    """
    def __init__(self, diagnostic_council):
        self.context = None
        self.is_fail = None

        self._append_mode = None
        self._diagnostic_council = diagnostic_council
        self._diagnostic_id = None
        self._diagnostic_reports = list()
        self._enabled = False

    @property
    def enabled(self):
        """
        Включенный объект диагностирует выбрасываемые функцией исключения и
        сохраняет эту или добавленную вручную диагностику в БД.

        Отключенной объект никак не влияет на ход выполнения функции,
        игнорирует все исключения и добавляемую вручную диагностику.
        """
        return self._enabled

    def enable(self, diagnostic_id, append=True):
        """
        Включить сбор диагностики. По-умолчанию она выключена.

        Входные параметры

            append -- признак, что нужно сохранить новые записи о проблемах, не
                удаляя старых.

            diagnostic_id -- идентификатор под которым будет сохранена вся
                собранная диагностика.
        """
        self._enabled = True

        self._append_mode = append
        self._diagnostic_id = diagnostic_id

    def build_manual_report(self, failure_diagnostics):
        """
        Добавить диагностику вручную

        Входные параметры

            failure_diagnostics -- объект с диагностикой
        """
        self._diagnostic_reports.append(failure_diagnostics)

    def diagnose(self, func, *args, **kwargs):
        """
        Запускает данную функцию под своим присмотром, а после её завершения
        сохраняет собранную диагностику в БД.
        """
        try:
            retval = func(*args, **kwargs)

        except Exception as e:
            if not self.enabled:
                raise
            return self._finish(e, None)

        else:
            if not self.enabled:
                return retval
            return self._finish(None, retval)

    def _finish(self, exception, retval):
        # Завершает диагностику проблемы и сохраняет её в БД
        # Если exception is not None, то нужно вызывать в скоупе блока except,
        # чтобы можно было перевыбросить исключение не потеряв traceback.
        self.is_fail = True

        if isinstance(exception, StopDiagnosticsException):
            processed, err = failsafe(self._process_exception, exception.cause)
            if err or not processed:
                retval = exception.on_fail_return

        elif isinstance(exception, Exception):
            processed, err = failsafe(self._process_exception, exception)
            if err or not processed:
                raise

        elif exception is None:
            processed, err = failsafe(self._process_ok)
            if not err and processed:
                self.is_fail = False

        else:
            raise NotImplementedError()
        return retval

    def _process_exception(self, exception):
        report = self._build_report_on_exception(exception)
        if report:
            self._save_collected_reports()
            return True

    def _process_ok(self):
        self._save_collected_reports()
        return True

    def _save_collected_reports(self):
        if self._diagnostic_reports or not self._append_mode:
            save_failure_diagnostics(
                self._diagnostic_id,
                self._diagnostic_reports,
                self._append_mode,
            )

    def _build_report_on_exception(self, exception):
        report = self._diagnostic_council.diagnose(exception, self.context)
        if report:
            self.build_manual_report(report)
            return report


class StopDiagnosticsException(Exception):
    """
    Эффект для DiagnosticManager от выброса этого исключение похож на эффект
    от выброса исключения cause, но отличается когда, DiagnosticManager не смог
    собрать диагностику о cause или сохранить её. В таком случае
    DiagnosticManager не перевыбрасывает cause, а делает признак is_fail
    равным True и возвращает в вызвавшую программу значение on_fail_return.
    """
    def __init__(self, cause, on_fail_return=None):
        self.cause = cause
        self.on_fail_return = on_fail_return


class DiagnosticInternalBrokerHandlerV1Mixin(object):
    """
    Добавляет возможность собрать диагностику о исключениях или других
    необычных ситуациях, которые встретятся во время выполнения запроса.

    В момент выполнения запроса разработчику доступен атрибут
    diagnostic_manager, через который можно управлять сбором диагностики.

    После выполнения запроса собранная диагностика сохраняется в БД.

    Если на момент выброса исключения или завершения запроса
    diagnostic_manager.enabled, то он проведёт диагностику исключения и сохранит
    её в БД. Если запрос завершился выбросом исключения, то ручка вернёт
    compose_diagnostic_response. Если запрос завершился без исключений, то ручка
    вернёт такой же ответ, какой бы она вернула, если бы diagnostic_manager был
    отключен.

    Если diagnostic_manager не смог диагностировать данное исключение, или
    сохранить диагностику, то данное исключение будет перевыброшено и ручка
    вернёт такой же отказ, какой бы она вернула, если бы diagnostic_manager был
    отключен.

    Если запрос завершился без исключений, но diagnostic_manager не смог
    сохранить записи о других проблемах, ручка вернёт compose_diagnostic_response.
    """
    def get(self):
        return self.diagnostic_get()

    def diagnostic_get(self):
        self.diagnostic_manager = self.build_diagnostic_manager()
        response = self.diagnostic_manager.diagnose(super(DiagnosticInternalBrokerHandlerV1Mixin, self).get)

        if self.diagnostic_manager.enabled and self.diagnostic_manager.is_fail:
            if response is not None:
                self.response = response
                return self.response

            self.response.data = self.compose_diagnostic_response()
            return self.response

        return response

    def compose_diagnostic_response(self):
        raise NotImplementedError()

    def build_diagnostic_manager(self):
        raise NotImplementedError()
