# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from functools import partial
import json
import logging
from os import path

from flask import request
from furl import furl
from passport.backend.core.cookies.utils import dump_cookie
from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.broker.binding import (
    bind_social_userinfo,
    BindingData,
)
from passport.backend.social.broker.communicators.communicator import Communicator
from passport.backend.social.broker.cookies import (
    build_json_secure_cookie,
    decrypt_cookie,
)
from passport.backend.social.broker.error_handler import process_error
from passport.backend.social.broker.exceptions import (
    AuthorizationRequiredError,
    CommunicationFailedError,
    DatabaseFailedError,
    GrantsMissingError,
    OAuthTokenInvalidError,
    SessionInvalidError,
    SocialBrokerError,
    TaskNotFoundError,
    UserDeniedError,
)
from passport.backend.social.broker.handlers.args_processor import ArgsProcessor
from passport.backend.social.broker.misc import get_grants_config
from passport.backend.social.broker.social_userinfo import get_social_userinfo
from passport.backend.social.broker.statbox import to_statbox
from passport.backend.social.common.builders.blackbox import (
    Blackbox,
    BlackboxInvalidResponseError,
    BlackboxInvalidSessionidError,
    BlackboxOauthTokenInvalidError,
    BlackboxTemporaryError,
    BlackboxUnknownError,
    check_oauth_response,
    check_oauth_response_suitable_for_binding,
    check_session_cookie,
)
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.exception import (
    InvalidTokenProxylibError,
    NetworkProxylibError,
    ProviderRateLimitExceededProxylibError,
    ProviderTemporaryUnavailableProxylibError,
)
from passport.backend.social.common.grants import (
    check_any_of_grants,
    filter_allowed_grants,
    GrantsContext,
    GrantsMissingError as CommonGrantsMissingError,
)
from passport.backend.social.common.misc import (
    class_property,
    GraphiteMessageType,
    name_for_graph_log,
    trim_message,
    write_graph_log_message,
)
from passport.backend.social.common.profile import (
    BaseProfileNotAllowedProfileCreationError,
    DatabaseFailedProfileCreatetionError,
    get_profile,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.redis_client import (
    get_redis,
    RedisError,
)
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.task import (
    create_task,
    generate_task_id,
    InvalidTaskDataError,
    load_task_from_redis,
    save_task_to_redis,
    Task,
)
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.web_service import (
    ApplicationUnknownWebServiceError,
    DatabaseFailedWebServiceError,
    ProfileNotAllowedWebServiceError,
    ProviderFailedWebServiceError,
    ProviderTokenInvalidWebServiceError,
    RateLimitExceededWebServiceError,
    Response,
    SocialInternalHandlerV2,
)


logger = logging.getLogger(__name__)

STATUS_REDIRECT = 'redirect'


class Handler(object):
    """
    Базовый класс-обработчик запроса. Содержит общие методы, нужные для разных обработчиков.
    """
    COOKIE_PATH = 'dummy/%s'
    _response_class = Response

    @classmethod
    def as_view(cls):
        view_func = partial(cls._view_func, cls)
        view_func.__name__ = cls.__name__
        return view_func

    @staticmethod
    def _view_func(cls, *args, **kwargs):
        handler = cls(request)
        try:
            return handler.get(*args, **kwargs)
        except Exception as e:
            return process_error(e, handler)

    @class_property
    def method_name(cls):
        return request_ctx.handler_id

    def __init__(self, request):
        self.request = request

        self.redis_client = get_redis()
        self.blackbox = Blackbox(get_http_pool_manager())

        self.response = self._response_class(mimetype='application/json')

        self.task = Task()
        self._task_id = None
        self.processed_args = dict()

    def _get_consumer_from_query_or_data(self):
        consumer = self.request.args.get('consumer') or self.request.form.get('consumer')
        if consumer:
            consumer = consumer.strip()
        return consumer or None

    def _get_consumer_from_headers(self):
        consumer = self.request.header_consumer
        if consumer:
            consumer = consumer.strip()
        return consumer or None

    @cached_property
    def _grants_context(self):
        return GrantsContext(
            consumer_ip=self.consumer_ip,
            consumer=self.consumer,
            ticket_body=self.request.ticket_body,
        )

    def get(self):
        """
        Базовый метод обработки запроса, определяется в потомках.
        """
        raise NotImplementedError()  # pragma: no cover

    @property
    def task_id(self):
        return self._task_id

    @task_id.setter
    def task_id(self, value):
        self._task_id = request_ctx.task_id = value

    def init_new_task(self):
        self.task = create_task()
        self.task_id = self.task.task_id

    def save_task_to_cookie(self):
        task_json_data = self.task.dump_session_data()
        return self._set_cookie(
            key='track',
            value=task_json_data,
            encrypted=True,
            frontend_url=self.processed_args['frontend_url'],
        )

    def _set_cookie(self, key, value, encrypted, frontend_url):
        logger.debug(trim_message('Setting cookie %s=%s' % (key, str(value)), cut=False))

        cookie_domain, cookie_path = self.get_cookie_path(frontend_url)

        if encrypted:
            value = build_json_secure_cookie(value, expires_in=social_config.track_cookie_expiration_time)

        return dump_cookie(
            key=key,
            value=value,
            httponly=True,
            domain=cookie_domain,
            max_age=social_config.track_cookie_expiration_time,
            path=cookie_path,
            sync_expires=True,
            secure=True,
        )

    def burn_task_cookie(self):
        return self.burn_cookie('track')

    def burn_cookie(self, key):
        logger.debug('Deleting cookie %s' % key)

        frontend_url = self.processed_args['frontend_url']
        cookie_domain, cookie_path = self.get_cookie_path(frontend_url)

        return dump_cookie(
            key=key,
            value='',
            httponly=True,
            domain=cookie_domain,
            max_age=0,
            path=cookie_path,
            sync_expires=True,
            secure=True,
        )

    def get_cookie_path(self, frontend_url):
        frontend_url = furl(frontend_url)
        domain = frontend_url.netloc
        return domain, path.join(str(frontend_url.path), self.COOKIE_PATH % self.task_id)

    def get_cookie_from_post_parameters(self, key):
        """
        От фронтенда получаем некоторые куки в виде GET параметров, эти куки сериализованы и подписаны.
        Если сессия протухла - кидаем SessionInvalidError
        """
        logger.debug('Getting cookie "%s" from POST parameters' % key)
        data = self.request.form.get(key)
        if not data:
            return
        return decrypt_cookie(data)

    def compose_json_response(self, args=None):
        return json.dumps(args) + '\n'

    def generate_task_id(self):
        """
        Генерация уникального task_id.
        """
        task_id = generate_task_id()
        logger.debug('task_id has been generated: %s' % task_id)
        return task_id

    def create_redirect_url_for_communicator(self, task_id, frontend_url):
        url = "%s%s/callback" % (frontend_url, task_id)
        logger.debug('Redirect url has been created: "%s"' % url)
        return url

    def create_communicator(self):
        """
        Создаем класс коммуникатора, выбираем приложение исходя из указанного
        пользователем или дефолтное.
        """
        logger.debug('Creating a communicator')
        self.communicator = Communicator.create(
            app=self.processed_args['application'],
            display=self.processed_args['display'],
            ui_language=self.processed_args['ui_language'],
            experiments=self.processed_args.get('experiments'),
        )
        logger.debug('%s has been created', type(self.communicator).__name__)

    def load_task_from_cookie(self):
        """
        Загружаем в self.task данные по задаче.
        """
        logger.debug('Loading task data')

        task_json_data = self.get_cookie_from_post_parameters('track')
        logger.debug('Raw dumped cookie length: %s' % len(task_json_data or ''))

        try:
            self.task.parse_session_data(task_json_data)
        except InvalidTaskDataError as e:
            raise SessionInvalidError(str(e))

        if self.task_id != self.task.task_id:
            raise SessionInvalidError("Received cookie task_id doesn't match cookie's one")

    def assert_state(self, state_expected):
        if isinstance(state_expected, basestring):
            state_expected = [state_expected]

        if self.task.state not in state_expected:
            raise SessionInvalidError('Invalid task state. Found: "%s". Expected: "%s"' % (self.task.state, state_expected))

    def get_uid(self, user_ip, hostname, session_id, oauth_token):
        """
        Получить uid пользователя по его сессии (или токену).

        Возвращает None, когда сессия недействительна.
        """
        logger.info('Requesting UID')

        uid = None
        self.task.display_name = ''
        blackbox_args = dict(
            ip=user_ip,
            need_display_name=True,
            attributes=[],
            dbfields=[],
        )
        try:
            if oauth_token:
                logger.debug('Checking OAuth token')
                blackbox_response = self.blackbox.oauth(oauth_token=oauth_token, **blackbox_args)
                check_oauth_response(blackbox_response)
                check_oauth_response_suitable_for_binding(blackbox_response)
                uid = blackbox_response['uid']
                self.task.display_name = self.build_display_name(blackbox_response)
            elif session_id:
                logger.debug('Checking Session_id')
                blackbox_response = self.blackbox.sessionid(sessionid=session_id, host=hostname, **blackbox_args)
                check_session_cookie(blackbox_response)
                uid = blackbox_response['uid']
                self.task.display_name = self.build_display_name(blackbox_response)
        except (BlackboxInvalidSessionidError, BlackboxOauthTokenInvalidError):
            logger.info('Getting account failed: invalid credential')
        except BlackboxTemporaryError:
            write_graph_log_message(GraphiteMessageType.error, self.method_name, 'blackbox', 'network')
            raise CommunicationFailedError('Getting account failed: network fail')
        except (BlackboxInvalidResponseError, BlackboxUnknownError):
            raise CommunicationFailedError('Getting account failed: blackbox fail')

        logger.debug('UID=%s, display_name=%s' % (uid or '', self.task.display_name))
        return uid

    def build_display_name(self, userinfo):
        display_name = userinfo.get('display_name', dict())
        if 'social' in display_name:
            display_name['social'].pop('redirect_target', None)
        return display_name

    def save_task_to_redis(self):
        try:
            save_task_to_redis(self.redis_client, self.task_id, self.task)
        except RedisError:
            logger.error('Unable to write task data to Redis')
            raise DatabaseFailedError()

        to_statbox({
            'task_id': self.task_id,
            'request_id': self.request.id,
            'action': 'saved',
            'userid': self.task.profile.get('userid'),
            'elapsed_seconds': int(now.f() - self.task.created),
            'email': self.task.profile.get('email'),
        })

    def update_task_from_redis(self):
        task = load_task_from_redis(self.redis_client, self.task_id)
        if not task:
            raise TaskNotFoundError('Task not found')
        self.task.update(task)

    def get_access_token(self, frontend_url, scopes):
        logger.info('Getting an access token')
        access_token = self.communicator.get_access_token(
            callback_url=self.create_redirect_url_for_communicator(self.task_id, frontend_url),
            exchange=self.task.exchange,
            request_token=self.task.request_token,
            scopes=scopes,
        )
        self.communicator.check_nonce(access_token, self.task.nonce)
        self.task.access_token = access_token

    def get_authorization(self, frontend_hostname, session_id, oauth_token, require_auth):
        try:
            uid = self.get_uid(self.processed_args['user_ip'], frontend_hostname, session_id, oauth_token)
        except CommunicationFailedError:
            if require_auth:
                raise
            uid = None
            logger.warning('Getting account failed: network fail')

        if require_auth:
            if not uid:
                raise AuthorizationRequiredError('Authorization required')
            if self.task.uid and self.task.uid != uid:
                raise SessionInvalidError("Authorization at /start and /continue doesn't match")
        self.task.uid = uid

    def log_that_request_is_ok(self):
        pass

    def build_task_processor(self):
        return ArgsProcessor(
            args=self.task.start_args,
            form=self.task.start_args,
            log_actions=False,
        )

    def build_request_processor(self):
        return ArgsProcessor(
            self.request.args,
            self.request.form,
            log_actions=True,
        )

    def process_task_args(self, processor, tld):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        processor.process_consumer()
        processor.process_retpath()
        processor.fix_morda_retpath()
        processor.process_place()
        self.processed_args.update(processor.processed_args)

        processor.get_bool('return_brief_profile')
        processor.get_bool('require_auth')
        processor.get_bool('force_prompt')
        processor.process_display()
        processor.process_app_and_provider(tld)
        processor.process_scope()
        processor.process_sid()
        processor.process_login_hint()

        self.processed_args.update(processor.processed_args)

    def process_sessionid_args(self):
        processor = ArgsProcessor(
            args=self.request.args,
            form=self.request.form,
            log_actions=True,
        )

        processor.process_session_id()
        processor.process_user_ip()
        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        processor.process_yandexuid()

        self.processed_args.update(processor.processed_args)

    def process_yandex_token_args(self, processor):
        processor.process_yandex_token()
        processor.process_yandexuid()
        self.processed_args.update(processor.processed_args)

        request_processor = ArgsProcessor(
            args=self.request.args,
            form=self.request.form,
            log_actions=True,
        )
        request_processor.process_user_ip()
        self.processed_args.update(request_processor.processed_args)

    def task_can_be_retried(self):
        return self.task.start_args is not None

    @property
    def consumer_ip(self):
        return self.request.consumer_ip

    @property
    def consumer(self):
        return self._get_consumer_from_query_or_data()

    def consumer_has_grant(self, grant_name):
        grants_config = get_grants_config()
        grants_config.load()
        return bool(filter_allowed_grants(grants_config, self._grants_context, [grant_name]))

    def check_grant(self, grant_name):
        grants_config = get_grants_config()
        grants_config.load()
        try:
            check_any_of_grants(grants_config, self._grants_context, [grant_name])
        except CommonGrantsMissingError as e:
            logger.warning('Access denied (%s)' % str(e))
            raise GrantsMissingError(error=e)
        logger.info('Access to %s is granted for %s' % (grant_name, str(self._grants_context)))

    def should_frontend_passthrough_error(self, exception):
        if not self.task.passthrough_errors:
            return False

        ALLOWED_PASSTHROUGH_EXCEPTIONS = [
            AuthorizationRequiredError,
            UserDeniedError,
        ]
        passthrough_errors = set(e.lower() for e in self.task.passthrough_errors)
        for error in ALLOWED_PASSTHROUGH_EXCEPTIONS:
            if type(exception) is error and error.__name__.lower() in passthrough_errors:
                return True
        else:
            return False

    @property
    def task(self):
        return request_ctx.task

    @task.setter
    def task(self, value):
        request_ctx.task = value


class CallbackHandler(Handler):
    EXPECTED_STATE = None
    NEXT_STATE = None

    def build_continue_url(self):
        raise NotImplementedError()  # pragma: no cover

    def get(self, task_id):
        self.task_id = task_id
        self.load_task_from_cookie()
        self.process_callback_args()

        if self.find_better_handler_url_for_state():
            return self.compose_send_to_better_handler_for_state_response()

        self.assert_state(self.EXPECTED_STATE)
        self.log_that_request_is_ok()
        self.create_communicator()

        exchange = self.get_exchange_value_from_callback()

        self.task.state = self.NEXT_STATE
        self.task.exchange = exchange
        task_cookie = self.save_task_to_cookie()

        location = self.build_continue_url()
        self.response.data = self.compose_json_response(
            dict(
                task_id=self.task_id,
                request_id=self.request.id,
                location=location,
                cookies=[task_cookie],
            ),
        )

        logger.info('Redirecting to "%s"' % location)
        return self.response

    def get_exchange_value_from_callback(self):
        """
        Проверить наличие exchange value в параметрах вызова.
        """
        logger.debug('Checking callback for errors')
        if self.task.request_token:
            request_token = self.task.request_token.get('value')
        else:
            request_token = None

        query = self.get_callback_query()

        try:
            self.communicator.has_error_in_callback(query, request_token_value=request_token)
        except SocialBrokerError:
            self.log_that_getting_exchange_value_failed()
            raise

        logger.debug('Getting exchange value')
        try:
            exchange = self.communicator.get_exchange_value_from_callback(query)
        except SocialBrokerError:
            self.log_that_getting_exchange_value_denied()
            raise

        self.log_that_getting_exchange_value_ok()
        return exchange

    def get_callback_query(self):
        return self.request.args

    def log_that_getting_exchange_value_ok(self):
        to_statbox(dict(self._get_statbox_args(), status='ok'))

    def log_that_getting_exchange_value_failed(self):
        to_statbox(dict(self._get_statbox_args(), status='error'))

    def log_that_getting_exchange_value_denied(self):
        to_statbox(dict(self._get_statbox_args(), status='denied'))

    def _get_statbox_args(self):
        return dict(
            task_id=self.task_id,
            request_id=self.request.id,
            action='callbacked',
        )

    def find_better_handler_url_for_state(self):
        pass

    def compose_send_to_better_handler_for_state_response(self):
        redirect_response = RedirectResponse.from_handler(self)
        redirect_response.location = self.find_better_handler_url_for_state()
        logger.info('Redirecting to "%s"' % redirect_response.location)
        return redirect_response.set()

    def process_callback_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        task_processor = self.build_task_processor()

        task_processor.process_consumer()
        task_processor.process_retpath()
        task_processor.fix_morda_retpath()
        task_processor.process_place()

        self.processed_args.update(task_processor.processed_args)

        request_processor = self.build_request_processor()

        request_processor.process_frontend_url()
        request_processor.process_hostname_and_tld(request_processor.processed_args['frontend_url'])
        request_processor.create_retry_url(request_processor.processed_args['frontend_url'], self.task_id)

        self.processed_args.update(request_processor.processed_args)

        request_processor.process_ui_language()
        self.processed_args.update(request_processor.processed_args)

        self.process_task_args(task_processor, self.processed_args['tld'])


class InternalBrokerHandlerV2(SocialInternalHandlerV2):
    _response_class = Response

    @classmethod
    def as_view(cls):
        view_func = partial(cls._view_func, cls)
        view_func.__name__ = cls.__name__
        return view_func

    @staticmethod
    def _view_func(cls, *args, **kwargs):
        handler = cls(request)
        return handler.get(*args, **kwargs)

    def __init__(self, request):
        super(InternalBrokerHandlerV2, self).__init__(request=request)

    @class_property
    def method_name(cls):
        return request_ctx.handler_id

    def _get_grants_config(self):
        return get_grants_config()

    def _get_application_from_client_id(self, provider_code, client_id):
        provider_info = providers.get_provider_info_by_name(provider_code)
        if not provider_info:
            raise ApplicationUnknownWebServiceError()

        app = providers.get_application_by_provider_app_id(provider_info['id'], client_id)
        if not app:
            raise ApplicationUnknownWebServiceError()
        return app

    def _get_application_from_name(self, app_name):
        app = providers.get_application_by_name(app_name)
        if not app:
            raise ApplicationUnknownWebServiceError()
        return app

    def _sanitize_client_token(self, app, client_token):
        try:
            server_token, refresh_token = Communicator.create(app).sanitize_client_token(client_token)
        except (InvalidTokenProxylibError, OAuthTokenInvalidError):
            raise ProviderTokenInvalidWebServiceError()
        except ProviderTemporaryUnavailableProxylibError:
            raise ProviderFailedWebServiceError()
        except ProviderRateLimitExceededProxylibError:
            raise RateLimitExceededWebServiceError()
        return server_token, refresh_token

    def _get_social_userinfo(self, app, token, extra_userinfo=None):
        try:
            userinfo = get_social_userinfo(app, token, extra_userinfo=extra_userinfo, user_ip=self._user_ip)
        except (InvalidTokenProxylibError, OAuthTokenInvalidError):
            raise ProviderTokenInvalidWebServiceError()
        except ProviderTemporaryUnavailableProxylibError:
            raise ProviderFailedWebServiceError()
        except NetworkProxylibError:
            write_graph_log_message(GraphiteMessageType.error, request_ctx.handler_id, name_for_graph_log(app), 'network')
            raise ProviderFailedWebServiceError()
        except ProviderRateLimitExceededProxylibError:
            raise RateLimitExceededWebServiceError()
        return userinfo

    def _bind_by_token(self, account, app, token, extra_social_userinfo=None):
        userinfo = self._get_social_userinfo(app, token, extra_userinfo=extra_social_userinfo)
        self._bind_by_social_userinfo(account, userinfo, token)

    def _bind_by_social_userinfo(self, account, userinfo, token):
        binding_data = BindingData(
            token=token,
            consumer=self._consumer,
        )
        try:
            bind_social_userinfo(account, userinfo, binding_data)
        except BaseProfileNotAllowedProfileCreationError:
            raise ProfileNotAllowedWebServiceError()
        except DatabaseFailedProfileCreatetionError:
            raise DatabaseFailedWebServiceError()

    def _does_binding_exist(self, account, social_userinfo):
        task = Task.from_social_userinfo(social_userinfo, account.uid, BindingData())
        return get_profile(get_slave_engine(), task, account.uid) is not None


class RedirectResponse(object):
    def __init__(self):
        self._handler = None
        self.location = None

    @classmethod
    def from_handler(cls, handler):
        response = RedirectResponse()
        response._handler = handler
        return response

    def set(self):
        self._handler.response.data = self._handler.compose_json_response(
            dict(
                location=self.location,
                request_id=self._handler.request.id,
                status=STATUS_REDIRECT,
                task_id=self._handler.task_id,
            ),
        )
        return self._handler.response
