# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.broker.social_userinfo import get_social_userinfo
from passport.backend.social.common import (
    exception as common_exceptions,
    validators,
    web_service,
)
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.misc import split_scope_string
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.task import (
    create_task,
    save_task_to_redis,
)
from passport.backend.social.common.token.utils import find_all_tokens_for_account
from passport.backend.social.proxylib.refresh_token import filter_and_refresh_newest_token


class _CreateTaskForProfileForm(validators.Schema):
    application_name = validators.ApplicationName()
    scope = validators.Scope(if_missing=None)
    task_consumer = validators.Consumer()


class CreateTaskForProfile(InternalBrokerHandlerV2):
    basic_form = _CreateTaskForProfileForm
    required_grants = ['create-task-for-profile']

    def _process_request(self):
        self.get_application()
        self.get_account()
        self.find_token()
        self.create_task()
        self.response_values.update(task_id=self.task.task_id)

    def get_application(self):
        self.application = providers.get_application_by_name(self.form_values.get('application_name') or '')
        if not self.application:
            raise web_service.ProviderTokenNotFoundWebServiceError()

    def get_account(self):
        self.account = self._get_account_from_user_ticket()

    def find_token(self):
        tokens = find_all_tokens_for_account(
            self.account.uid,
            get_slave_engine(),
            application_ids=[self.application.identifier],
        )

        required_scope = (
            self.form_values.get('scope') or
            (split_scope_string(self.application.allowed_scope) if self.application.allowed_scope else list())
        )
        tokens = [t for t in tokens if t.has_every_scope(required_scope)]

        try:
            self.token = filter_and_refresh_newest_token(tokens)
        except common_exceptions.NotFound:
            raise web_service.ProviderTokenNotFoundWebServiceError()

    def create_task(self):
        try:
            userinfo = get_social_userinfo(self.token.application, self.token, user_ip=self._user_ip)
        except common_exceptions.InvalidTokenProxylibError:
            raise web_service.ProviderTokenNotFoundWebServiceError()

        self.task = create_task()
        self.task.access_token = self.token.to_dict_for_proxy()
        self.task.application = self.token.application
        self.task.consumer = self.form_values.get('task_consumer')
        self.task.finished = self.task.created
        self.task.profile = userinfo
        self.task.provider = self.token.application.provider
        self.task.uid = self.account.uid

        save_task_to_redis(get_redis(), self.task.task_id, self.task)
