# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import logging

from passport.backend.social.broker.handlers.base import Handler
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.misc import (
    format_traceback,
    GraphiteMessageType,
    trim_message,
    write_graph_log_message,
)
from passport.backend.social.common.multiprocessing import execute_multiple_methods
from passport.backend.social.common.profile import execute
from passport.backend.social.common.social_config import social_config
from passport.backend.social.proxylib import useragent


logger = logging.getLogger('social.broker.handlers')


class PingHandler(Handler):
    def get(self):
        write_graph_log_message(GraphiteMessageType.other, self.method_name)

        def append_error(error):
            errors.append(error)
            # Смысл пинга сообщить о здоровье приложения, не следует каждый раз
            # когда что-то не в порядке писать в лог с ошибками. Но для
            # расследования лучше всё-таки сохранить диагностику в информационном
            # журнале.
            logger.info('Error while pinging service %s: %s', error, format_traceback())

        check = self.request.args.get('check')
        if not check:
            self.response.data = 'Error: "check" parameter should be specified\n'
            self.response.status_code = 400
            return self.response

        try:
            fd = open(social_config.ping_file_path, 'r')
            fd.read(100)
            fd.close()
        except Exception:
            write_graph_log_message(GraphiteMessageType.other, 'ping_error', 'file')
            self.response.data = json.dumps({'errors': [social_config.ping_file_path]})
            self.response.status_code = 503
            return self.response

        check = check.split(',')
        errors = []
        jobs = []

        def _check_db():
            try:
                execute(get_slave_engine(), 'select 1;')
            except:
                append_error('db')

        def _check_frontend():
            try:
                response = useragent.execute_request(
                    method='GET',
                    url='http://127.0.0.1:3000/broker2/ping',
                    retries=2,
                    timeout=0.5,
                    service='frontend',
                )
                if response.decoded_data.lower() != 'pong':
                    raise Exception('Response != pong. %s' % trim_message(response.decoded_data))
            except:
                append_error('frontend')

        def _check_blackbox():
            try:
                uid = social_config.blackbox_availability_test_uid
                response = self.blackbox.userinfo(
                    uid=uid,
                    ip='127.0.0.1',
                    attributes=[],
                    dbfields=[],
                )
                if uid != response['uid']:
                    raise Exception('invalid uid: %s != %s' % (uid, response.uid))
            except:
                append_error('blackbox')

        def _check_social_api():
            try:
                response = useragent.execute_request(
                    method='GET',
                    url='http://127.0.0.1/api/ping',
                    headers={'Host': social_config.social_api_host},
                    retries=2,
                    timeout=0.5,
                    service='social_api',
                )
                if response.decoded_data.lower().strip() != 'pong':
                    raise Exception('Response != pong. %s' % response.decoded_data)
            except:
                append_error('social_api')

        if 'social_api' in check:
            jobs.append(_check_social_api)

        if 'db' in check:
            jobs.append(_check_db)

        if 'frontend' in check:
            jobs.append(_check_frontend)

        if 'blackbox' in check:
            jobs.append(_check_blackbox)

        execute_multiple_methods(jobs)

        if errors:
            for error in errors:
                write_graph_log_message(GraphiteMessageType.other, 'ping_error', error)

            self.response.data = json.dumps({'errors': errors})
            self.response.status_code = 503
            return self.response

        self.response.status_code = 200
        self.response.mimetype = 'text/plain'
        self.response.data = 'Pong\n'
        return self.response
