# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from functools import partial
import json
import logging

from passport.backend.social.broker.communicators.communicator import (
    AuthorizeOptions,
    OAuthCommunicator,
)
from passport.backend.social.broker.exceptions import UserRejectedBindError
from passport.backend.social.broker.handlers.base import (
    CallbackHandler as _BaseCallbackHandler,
    Handler,
)
from passport.backend.social.broker.handlers.profile import task_state
from passport.backend.social.broker.handlers.profile.base import (
    COOKIE_PATH,
    ProfileHandler,
)
from passport.backend.social.broker.statbox import to_statbox
from passport.backend.social.common.builders.passport import Passport
from passport.backend.social.common.chrono import now
from passport.backend.social.common.misc import trim_message
from passport.backend.social.common.multiprocessing import execute_multiple_methods
from passport.backend.social.common.useragent import get_http_pool_manager


logger = logging.getLogger(__name__)


def get_better_url_from_web_profile_handler_state(handler):
    """
    Подыскивает ручку, которая лучше подходит для состояния в котором находится
    вызвавший ручку пользователь.
    """
    if handler.task.state == task_state.PROFILE_R_TO_CONT and type(handler) is not ContinueHandler:
        return handler.build_continue_url()
    elif handler.task.state == task_state.PROFILE_R_TO_BIND and type(handler) is not BindHandler:
        return handler.build_bind_url()


class StartHandler(Handler):
    COOKIE_PATH = COOKIE_PATH

    def get(self, task_id=None):
        self.check_grant('authz-in-web')

        self.retry_mode = bool(task_id)

        if self.retry_mode:
            logger.debug('Retrying to start...')
            self.task_id = task_id
            self.load_task_from_cookie()
            if self.task.in_redis:
                self.update_task_from_redis()

            self.assert_state([
                task_state.PROFILE_R_TO_SOC,
                task_state.PROFILE_R_TO_CONT,
                task_state.PROFILE_R_TO_BIND,
                task_state.PROFILE_CAPTCHA,
            ])

            self.process_retry_args()
        else:
            self.init_new_task()
            self.process_start_args()
            self.task.start_args = dict(self.request.args)

        self.fill_task_with_processed_args()

        self.task.state = task_state.PROFILE_R_TO_SOC

        self.log_that_request_is_ok()
        self.write_to_statbox()

        if self.processed_args['require_auth']:
            self.get_authorization(
                self.processed_args['hostname'],
                self.processed_args['Session_id'],
                self.processed_args['oauth_token'],
                require_auth=True,
            )

        self.create_communicator()

        logger.debug('New task has been generated: %s' % trim_message(self.task.dump_session_data()))

        # та же строка, но, возможно, с другим разделителем
        scope_string = self.communicator.get_scope(self.processed_args['scope'])
        self.task.scope = scope_string

        nonce = self.communicator.build_nonce()
        self.task.nonce = nonce

        callback_url = self.build_callback_url()
        self.task.callback_url = callback_url

        if isinstance(self.communicator, OAuthCommunicator):
            auth_redirect_link, request_token = self.communicator.get_authorize_url(AuthorizeOptions(
                callback_url,
                scope=scope_string,
            ))
            self.task.request_token = json.loads(request_token.to_json())
        else:
            auth_redirect_link = self.communicator.get_authorize_url(AuthorizeOptions(
                callback_url,
                scope=scope_string,
                force_prompt=self.processed_args['force_prompt'],
                login_hint=self.processed_args['login_hint'],
                nonce=nonce,
            ))

        logger.debug('redirect to = %s' % auth_redirect_link)

        task_cookie = self.save_task_to_cookie()
        self.save_task_to_redis()

        self.response.data = self.compose_json_response({
            'request_id': self.request.id,
            'provider': self.processed_args['provider'],
            'task_id': self.task_id,
            'retry_url': self.processed_args['retry_url'],
            'location': auth_redirect_link,
            'cookies': [task_cookie],
        })

        return self.response

    def write_to_statbox(self):
        statbox_data = {
            'task_id': self.task_id,
            'request_id': self.request.id,
            'session_id_passed': bool(self.processed_args['Session_id']),
            'token_passed': bool(self.processed_args['oauth_token']),
            'ip': self.processed_args['user_ip'],
            'yandexuid': self.processed_args['yandexuid'],
        }
        if self.retry_mode:
            statbox_data['action'] = 'restarted'
        else:
            for key in ['retpath', 'require_auth', 'consumer', 'sid']:
                statbox_data[key] = self.processed_args[key]

            statbox_data.update({
                'action': 'started',
                'require_auth': bool(self.processed_args['require_auth']),
                'scope': ','.join(self.processed_args['scope']),
                'provider': self.processed_args['provider']['name'],
                'application': self.processed_args['application'].name,
                'place': self.processed_args['place'] or 'query',
            })

        to_statbox(statbox_data)

    def build_callback_url(self):
        return '%s%s/callback' % (self.processed_args['frontend_url'], self.task_id)

    def process_start_args(self):
        processor = self.build_request_processor()

        processor.fix_not_yandex_facebook_app()
        processor.process_consumer()
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        processor.process_retpath()
        processor.fix_morda_retpath()
        processor.process_place()
        processor.process_passthrough_errors()
        processor.process_experiments()
        self.processed_args.update(processor.processed_args)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        processor.create_retry_url(processor.processed_args['frontend_url'], self.task_id)
        self.processed_args.update(processor.processed_args)

        self.process_task_args(processor, processor.processed_args['tld'])

        self.process_sessionid_args()
        self.process_yandex_token_args(processor)

        processor.process_pkce(required=False)
        processor.process_ui_language()
        self.processed_args.update(processor.processed_args)

    def process_retry_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        task_processor = self.build_task_processor()

        task_processor.process_consumer()
        task_processor.process_retpath()
        task_processor.fix_morda_retpath()
        task_processor.process_place()
        task_processor.process_passthrough_errors()
        task_processor.process_experiments()

        self.processed_args.update(task_processor.processed_args)

        task_processor.process_pkce(required=False)
        self.processed_args.update(task_processor.processed_args)

        request_processor = self.build_request_processor()

        request_processor.process_frontend_url()
        request_processor.process_hostname_and_tld(request_processor.processed_args['frontend_url'])
        request_processor.create_retry_url(request_processor.processed_args['frontend_url'], self.task_id)

        self.processed_args.update(request_processor.processed_args)

        self.process_sessionid_args()
        self.process_yandex_token_args(task_processor)

        self.process_task_args(task_processor, self.processed_args['tld'])

        request_processor.process_ui_language()
        self.processed_args.update(request_processor.processed_args)

    def fill_task_with_processed_args(self):
        self.task.start_args['token'] = self.processed_args['oauth_token']
        self.task.consumer = self.processed_args['consumer']
        self.task.sid = self.processed_args['sid']
        self.task.provider = self.processed_args['provider']
        self.task.application = self.processed_args['application']
        self.task.retpath = self.processed_args['retpath']
        self.task.place = self.processed_args['place']
        self.task.code_challenge = self.processed_args['code_challenge']
        self.task.code_challenge_method = self.processed_args['code_challenge_method']
        self.task.yandexuid = self.processed_args['yandexuid']
        self.task.passthrough_errors = self.processed_args['passthrough_errors']
        self.task.experiments = self.processed_args['experiments']

    @property
    def consumer(self):
        return self._get_consumer_from_headers()


class RetryHandler(StartHandler):
    pass


class CallbackHandler(_BaseCallbackHandler):
    EXPECTED_STATE = task_state.PROFILE_R_TO_SOC
    NEXT_STATE = task_state.PROFILE_R_TO_CONT
    COOKIE_PATH = COOKIE_PATH

    def get(self, *args, **kwargs):
        self.check_grant('authz-in-web')
        return super(CallbackHandler, self).get(*args, **kwargs)

    def build_continue_url(self):
        return '%s%s/continue' % (self.processed_args['frontend_url'], self.task_id)

    def build_bind_url(self):
        return '%s%s/bind' % (self.processed_args['frontend_url'], self.task_id)

    @property
    def consumer(self):
        return self._get_consumer_from_headers()

    def find_better_handler_url_for_state(self):
        return get_better_url_from_web_profile_handler_state(self)


class ContinueHandler(ProfileHandler):
    def get(self, task_id):
        self.check_grant('authz-in-web')

        self.task_id = task_id
        self.load_task_from_cookie()
        self.process_continue_args()
        self.assert_state([task_state.PROFILE_R_TO_CONT])
        self.log_that_request_is_ok()

        self.create_communicator()

        if self.task.in_redis:
            self.update_task_from_redis()
        self.task.yandexuid = self.processed_args['yandexuid']

        execute_multiple_methods([
            self.get_access_token_and_profile,
            partial(
                self.get_authorization,
                self.processed_args['hostname'],
                self.processed_args['Session_id'],
                self.processed_args['oauth_token'],
                self.processed_args['require_auth'],
            ),
        ])

        self.task.finished = now.f()
        self.save_task_to_redis()

        if self.processed_args['require_auth']:
            if self.should_confirm_bind():
                self.task.state = task_state.PROFILE_R_TO_BIND
                self.compose_send_to_bind_response(
                    self.processed_args['frontend_url'],
                    self.processed_args['retpath'],
                    self.processed_args['place'],
                    self.processed_args['provider'],
                )
                return self.response
            else:
                self.bind_profile(has_profile=True, sid=self.processed_args['sid'])

        self.compose_profile_response(
            self.processed_args['return_brief_profile'],
            self.processed_args['retpath'],
            self.processed_args['place'],
            keep_task_cookie=True,
        )
        return self.response

    def process_continue_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        task_processor = self.build_task_processor()

        task_processor.process_consumer()
        task_processor.process_retpath()
        task_processor.fix_morda_retpath()
        task_processor.process_place()

        self.processed_args.update(task_processor.processed_args)

        request_processor = self.build_request_processor()

        request_processor.process_frontend_url()
        request_processor.process_hostname_and_tld(request_processor.processed_args['frontend_url'])
        request_processor.create_retry_url(request_processor.processed_args['frontend_url'], self.task_id)

        self.processed_args.update(request_processor.processed_args)

        request_processor.process_ui_language()
        self.processed_args.update(request_processor.processed_args)

        self.process_sessionid_args()
        self.process_yandex_token_args(task_processor)

        self.process_task_args(task_processor, self.processed_args['tld'])

    @property
    def consumer(self):
        return self._get_consumer_from_headers()


class BindHandler(ProfileHandler):
    def get(self, task_id):
        self.check_grant('authz-in-web-bind')

        self.task_id = task_id
        self.load_task_from_cookie()
        self.process_bind_args()
        self.assert_state(task_state.PROFILE_R_TO_BIND)
        self.log_that_request_is_ok()

        self.create_communicator()

        if not self.processed_args['allow']:
            raise UserRejectedBindError()

        self.get_authorization(
            self.processed_args['hostname'],
            self.processed_args['Session_id'],
            self.processed_args['oauth_token'],
            require_auth=True,
        )
        self.update_task_from_redis()
        self.bind_profile(has_profile=False, sid=self.processed_args['sid'])

        self.compose_profile_response(
            self.processed_args['return_brief_profile'],
            self.processed_args['retpath'],
            self.processed_args['place'],
            keep_task_cookie=True,
        )
        self.send_notifications()
        return self.response

    def send_notifications(self):
        passport = Passport(get_http_pool_manager())
        passport.send_account_modification_notifications(
            self.processed_args['hostname'],
            self.processed_args['user_ip'],
            self.task.uid,
            'social_add',
            social_provider=self.processed_args['provider']['name'],
        )

    def process_bind_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        task_processor = self.build_task_processor()

        task_processor.process_consumer()
        task_processor.process_retpath()
        task_processor.fix_morda_retpath()
        task_processor.process_place()

        self.processed_args.update(task_processor.processed_args)

        request_processor = self.build_request_processor()

        request_processor.process_frontend_url()
        request_processor.process_hostname_and_tld(request_processor.processed_args['frontend_url'])
        request_processor.create_retry_url(request_processor.processed_args['frontend_url'], self.task_id)

        request_processor.get_bool('allow', from_form=True)
        request_processor.process_ui_language()

        self.processed_args.update(request_processor.processed_args)

        self.process_sessionid_args()
        self.process_yandex_token_args(task_processor)

        self.process_task_args(task_processor, self.processed_args['tld'])

    @property
    def consumer(self):
        return self._get_consumer_from_headers()
