# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from functools import partial
import logging

from passport.backend.social.broker.binding import (
    bind_social_userinfo_by_task,
    BindingData,
)
from passport.backend.social.broker.communicators.communicator import Communicator
from passport.backend.social.broker.exceptions import (
    DatabaseFailedError,
    ProfileNotAllowedError,
    SessionInvalidError,
)
from passport.backend.social.broker.handlers.base import Handler
from passport.backend.social.broker.misc import generate_retpath
from passport.backend.social.broker.social_userinfo import get_social_userinfo
from passport.backend.social.broker.statbox import to_statbox
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.multiprocessing import execute_multiple_methods
from passport.backend.social.common.profile import (
    BaseProfileCreationError,
    BaseProfileNotAllowedProfileCreationError,
    DatabaseFailedProfileCreatetionError,
    get_profile,
)
from passport.backend.social.common.refresh_token.utils import build_refresh_token_from_token_data
from passport.backend.social.common.token.domain import Token
from passport.backend.social.common.token.utils import build_token_dict_for_proxy
from passport.backend.utils.string import smart_text


COOKIE_PATH = '%s/'

logger = logging.getLogger(__name__)


class ProfileHandler(Handler):
    COOKIE_PATH = COOKIE_PATH

    def get_profile_information(self, extra_userinfo=None):
        token = Token(
            application_id=self.communicator.app.identifier,
            value=self.task.access_token['value'],
            secret=self.task.access_token.get('secret'),
            scopes=self.task.access_token.get('scope'),
            expired=self.task.access_token.get('expires'),
            created=None,
            verified=None,
            confirmed=None,
        )
        self.task.profile = get_social_userinfo(
            self.communicator.app,
            token,
            extra_userinfo=extra_userinfo,
            should_retry_on_invalid_token=True,
        )

    def bind_profile(self, has_profile, sid):
        logger.info('Saving profile to database. uid=%s' % self.task.uid)
        binding_data = BindingData(sid=sid)
        try:
            bind_social_userinfo_by_task(self.task, binding_data, has_profile)
        except DatabaseFailedProfileCreatetionError():
            raise DatabaseFailedError()
        except BaseProfileNotAllowedProfileCreationError:
            raise ProfileNotAllowedError()
        except BaseProfileCreationError:
            raise SessionInvalidError()

    def should_confirm_bind(self):
        # Если профиль уже есть - не отправляем на подтверждение
        logger.debug('Checking profile existence...')
        profile = get_profile(get_slave_engine(), self.task, self.task.uid)
        profile_id = profile.profile_id if profile else None
        logger.info('Current profile in mysql: %s', profile_id)

        to_statbox({
            'task_id': self.task_id,
            'request_id': self.request.id,
            'action': 'bind_status',
            'uid': self.task.uid,
            'auth_source': 'token' if self.processed_args['oauth_token'] else 'session_id',
            'requested': not bool(profile),
        })
        return profile is None

    def get_access_token_and_profile(self):
        if self.task.access_token is None:
            self.get_access_token(self.processed_args['frontend_url'], self.processed_args['scope'])

        server_token = Token.from_dict_for_proxy(self.task.access_token)
        refresh_token = build_refresh_token_from_token_data(self.task.access_token)

        def _sanitize_server_token():
            self.task.access_token = self.communicator.sanitize_server_token(server_token, refresh_token)

        execute_multiple_methods([
            _sanitize_server_token,
            partial(
                self.get_profile_information,
                extra_userinfo=self.task.access_token.get('extra_userinfo')
            ),
        ])

    def compose_send_to_bind_response(self, frontend_url, retpath, place, provider):
        # отправляем на подтверждение привязки
        location = frontend_url + self.task_id + '/bind'
        cancel_url = generate_retpath(
            retpath,
            place,
            'error',
        )

        name = '%s %s' % (self.task.profile.get('firstname', ''), self.task.profile.get('lastname', ''))
        name = ' '.join(name.split())

        task_cookie = self.save_task_to_cookie()

        self.response.data = self.compose_json_response({
            'bind_url': location,
            'cancel_url': cancel_url,
            'state': 'bind',
            'display_name': self.task.display_name,
            'profile': {
                'name': name,
                'userid': smart_text(self.task.profile.get('userid') or ''),
                'username': self.task.profile.get('username', ''),
            },
            'provider': provider,
            'cookies': [task_cookie],
        })
        logger.info('Redirecting to "%s"' % location)

    def compose_profile_response(
        self,
        return_brief_profile,
        retpath,
        place,
        keep_task_cookie=False,
    ):
        logger.info('Generating success response')

        response = dict()

        additional_args = {}
        if return_brief_profile and self.task.profile_id:
            additional_args['profile.profile_id'] = self.task.profile_id

        response.update(
            location=generate_retpath(
                retpath,
                place,
                'ok',
                task_id=self.task_id,
                additional_args=additional_args,
            ),
        )

        cookies = list()
        if keep_task_cookie:
            cookies.append(self.save_task_to_cookie())
        else:
            cookies.append(self.burn_task_cookie())
        response.update(cookies=cookies)

        self.response.data = self.compose_json_response(response)


class NativeProfileHandler(ProfileHandler):
    def get_profile_native(self, app, client_token):
        communicator = Communicator.create(app)
        server_token, refresh_token = communicator.sanitize_client_token(client_token)
        self.task.access_token = build_token_dict_for_proxy(server_token, refresh_token)
        userinfo = communicator.client_token_to_social_userinfo(client_token)
        self.task.profile = get_social_userinfo(app, server_token, userinfo)
        self.task.finished = now.f()
        self.save_task_to_redis()
        to_statbox({
            'task_id': self.task_id,
            'request_id': self.request.id,
            'action': 'task_by_token',
            'provider': app.provider and app.provider.get('name'),
            'application': app.name,
            'scope': ','.join(server_token.scopes),
            'uid': self.task.uid,
        })
