# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.broker.misc import invalidate_billing_cache
from passport.backend.social.broker.statbox import StatboxLogger
from passport.backend.social.common import validators
from passport.backend.social.common.providers.Kinopoisk import Kinopoisk
from passport.backend.social.common.token.domain import Token


logger = logging.getLogger(__name__)


class _BindKinopoiskAccountByTokenForm(validators.Schema):
    kinopoisk_session_id = validators.SessionId()
    kinopoisk_username = validators.Name()


class BindKinopoiskAccountByTokenHandler(InternalBrokerHandlerV2):
    """
    Привязать кинопоисковый аккаунт к яндексовому.
    """
    basic_form = _BindKinopoiskAccountByTokenForm
    required_grants = ['bind-kinopoisk-account-by-token']

    def _process_request(self):
        account = self._get_account_from_token()
        app = self._get_application_from_client_id(
            provider_code=Kinopoisk.code,
            client_id='kinopoisk-yandex',
        )
        token = Token(
            application_id=app.identifier,
            value=self.form_values['kinopoisk_session_id'],
        )
        username = self.form_values['kinopoisk_username']
        extra_social_userinfo = {'username': username} if username else {}
        kinopoisk_userinfo = self._get_social_userinfo(app, token, extra_social_userinfo)
        self._bind_by_social_userinfo(account, kinopoisk_userinfo, token)

        self._statbox.log(action='update_account_yandex_bindings', uid=account.uid)
        invalidate_billing_cache(account.uid)
        self._statbox.log(action='update_account_yandex_bindings', uid=kinopoisk_userinfo['userid'])
        invalidate_billing_cache(kinopoisk_userinfo['userid'])

    @cached_property
    def _statbox(self):
        return StatboxLogger(
            consumer=self._consumer,
            ip=self._user_ip,
        )


class _BindKinopoiskAccountBySessionIdForm(validators.Schema):
    yandex_session_id = validators.SessionId()
    kinopoisk_session_id = validators.SessionId()
    kinopoisk_username = validators.Name()


class BindKinopoiskAccountBySessionIdHandler(InternalBrokerHandlerV2):
    """
    Привязать кинопоисковый аккаунт к яндексовому по session_id яндексовского и Кинопоискового аккаунтов.
    """
    basic_form = _BindKinopoiskAccountBySessionIdForm
    required_grants = ['bind-kinopoisk-account-by-session-id']

    def _process_request(self):
        account = self._account_getter.get_account_from_session_id(
            self.form_values['yandex_session_id'],
            user_ip=self._user_ip,
            host='kinopoisk.ru',
        )
        app = self._get_application_from_client_id(
            provider_code=Kinopoisk.code,
            client_id='kinopoisk-yandex',
        )
        token = Token(
            application_id=app.identifier,
            value=self.form_values['kinopoisk_session_id'],
        )
        username = self.form_values['kinopoisk_username']
        extra_social_userinfo = {'username': username} if username else {}

        kinopoisk_userinfo = self._get_social_userinfo(app, token, extra_social_userinfo)
        self._bind_by_social_userinfo(account, kinopoisk_userinfo, token)

        self._statbox.log(action='update_account_yandex_bindings', uid=account.uid)
        invalidate_billing_cache(account.uid)
        self._statbox.log(action='update_account_yandex_bindings', uid=kinopoisk_userinfo['userid'])
        invalidate_billing_cache(kinopoisk_userinfo['userid'])

    @cached_property
    def _statbox(self):
        return StatboxLogger(
            consumer=self._consumer,
            ip=self._user_ip,
        )
