# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core import validators as passport_validators
from passport.backend.core.types.phone_number.phone_number import get_alt_phone_numbers_of_phone_number
from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.broker.binding import log_bind_profile
from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.broker.misc import invalidate_billing_cache
from passport.backend.social.broker.statbox import StatboxLogger
from passport.backend.social.common import validators
from passport.backend.social.common.builders.blackbox import Blackbox
from passport.backend.social.common.builders.passport import (
    Passport,
    PassportAccountDisabledError,
    PassportAccountNotFoundError,
    PassportPermanentError,
    PassportTemporaryError,
    PassportTrackInvalidStateError,
    PassportTrackNotFoundError,
    PassportUserNotVerifiedError,
)
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.profile import (
    BaseProfileNotAllowedProfileCreationError,
    ProfileCreator,
)
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.social_logging import (
    BindingCreatedStatboxEvent,
    BindingLogger,
)
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.web_service import (
    PassportFailedWebServiceError,
    ProfileNotAllowedWebServiceError,
)


logger = logging.getLogger(__name__)


class _BindPhonishAccountByTrackFormV2(validators.Schema):
    uid = passport_validators.Uid()
    track_id = passport_validators.TrackId()


class BindPhonishAccountByTrackHandlerV2(InternalBrokerHandlerV2):
    """
    По треку с подтверждённым номером связывает произвольный аккаунт и
    действующего фониша. Основной номер аккаунт, номер фониша и подтверждённый
    в треке номер должны совпадать.
    """
    basic_form = _BindPhonishAccountByTrackFormV2
    required_grants = ['bind-phonish-account-by-track']

    def __init__(self, request):
        super(BindPhonishAccountByTrackHandlerV2, self).__init__(request)

        self.master_account = None
        self.phonish = None

        # Объект ProfileCreator для связывания аккаунта и фониша
        self.profile_creator = None

        self.blackbox = Blackbox(get_http_pool_manager())
        self.allow_device_id_check = social_config.account_binding_allow_device_id_check

    def _process_request(self):
        self.load_master_account_and_phonish(
            self.form_values.get('uid'),
            self.form_values.get('track_id'),
        )

        phone_number = self.get_phonish_phone_number()
        if not self.does_master_secure_phone_equal(phone_number):
            logger.debug('Auth to %s with %s not allowed' % (self.master_account.uid, phone_number))
            raise ProfileNotAllowedWebServiceError()

        self.build_phonish_profile_creator()

        self.response_values.update(
            uid=self.master_account.uid,
            phonish_uid=self.phonish.uid,
        )
        if self.is_phonish_bound_to_master_account():
            self.response_values.update(old=True)
        else:
            self.bind_phonish_to_master_account()

    def load_master_account_and_phonish(self, master_uid, track_id,):
        passport = Passport(get_http_pool_manager())
        try:
            response = passport.get_phonish_uid_by_phone(
                track_id=track_id,
                user_ip=self._user_ip,
                use_device_id=self.allow_device_id_check,
            )
            phonish_uid = response.get('uid')
        except PassportUserNotVerifiedError:
            logger.debug('Phone not verified: %s' % track_id)
            raise ProfileNotAllowedWebServiceError()
        except PassportAccountDisabledError:
            logger.debug('Related phonish disabled: %s' % track_id)
            raise ProfileNotAllowedWebServiceError()
        except PassportAccountNotFoundError:
            logger.debug('Related phonish not found')
            raise ProfileNotAllowedWebServiceError()
        except PassportTemporaryError:
            raise PassportFailedWebServiceError()
        except PassportTrackNotFoundError:
            logger.debug('Track not found: %s' % track_id)
            raise ProfileNotAllowedWebServiceError()
        except PassportTrackInvalidStateError:
            logger.debug('Track state is not valid: %s' % track_id)
            raise ProfileNotAllowedWebServiceError()
        except PassportPermanentError as e:
            logger.error('Permanent Passport error: %s' % str(e))
            raise PassportFailedWebServiceError()

        accounts, unknown_uids = self._get_many_accounts_from_uids([master_uid, phonish_uid])
        if unknown_uids:
            logger.debug('Some accounts not found: %s' % ', '.join(map(str, sorted(unknown_uids))))
            raise ProfileNotAllowedWebServiceError()
        if accounts[0].uid == master_uid:
            self.master_account, self.phonish = accounts
        elif accounts[1].uid == master_uid:
            self.phonish, self.master_account = accounts
        else:
            raise NotImplementedError()  # pragma: no cover

        if not self.master_account.is_enabled:
            logger.debug('Master account is disabled: %s' % self.master_account.uid)
            raise ProfileNotAllowedWebServiceError()

    def get_phonish_phone_number(self):
        phones = self.phonish.phones.all()
        for phone in phones.itervalues():
            return phone.number

    def does_master_secure_phone_equal(self, phone_number):
        all_phone_numbers = [phone_number]
        all_phone_numbers.extend(get_alt_phone_numbers_of_phone_number(phone_number))
        return (
            self.master_account.phones.secure and
            any(p == self.master_account.phones.secure.number for p in all_phone_numbers)
        )

    def build_phonish_profile_creator(self):
        self.profile_creator = ProfileCreator(
            blackbox=self.blackbox,
            mysql_read=get_slave_engine(),
            mysql_write=get_master_engine(),
            social_userinfo=dict(provider=dict(code=Yandex.code), userid=str(self.phonish.uid)),
            timestamp=now.f(),
            token=None,
            uid=self.master_account.uid,
            yandexuid=None,
        )

    def is_phonish_bound_to_master_account(self):
        return bool(self.profile_creator.find_profile())

    def bind_phonish_to_master_account(self):
        try:
            self.profile_creator.check_profile_possible()
            profile_id = self.profile_creator.create()
            log_bind_profile(
                profile_created=True,
                profile_id=profile_id,
                task_id=None,
                uid=self.master_account.uid,
            )
        except BaseProfileNotAllowedProfileCreationError:
            logger.debug('Profile not allowed: (%s, %s)' % (self.master_account.uid, self.phonish.uid))
            raise ProfileNotAllowedWebServiceError()

        self._statbox.log(action='update_account_yandex_bindings', uid=self.master_account.uid)
        self._binding_log.log_event(
            BindingCreatedStatboxEvent(
                master_uid=self.master_account.uid,
                slave_userid=self.phonish.uid,
                provider_code=Yandex.code,
            ),
        )
        invalidate_billing_cache(self.master_account.uid)

    @cached_property
    def _statbox(self):
        return StatboxLogger(
            consumer=self._consumer,
            ip=self._user_ip,
        )

    @cached_property
    def _binding_log(self):
        return BindingLogger()
