# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.broker.binding import log_bind_profile
from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.broker.misc import invalidate_billing_cache
from passport.backend.social.broker.statbox import StatboxLogger
from passport.backend.social.common import validators
from passport.backend.social.common.builders.blackbox import Blackbox
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.profile import (
    BaseProfileNotAllowedProfileCreationError,
    create_profile,
)
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_logging import (
    BindingCreatedStatboxEvent,
    BindingLogger,
)
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.web_service import ProfileNotAllowedWebServiceError


logger = logging.getLogger(__name__)


class _BindPhonishAccountByUidForm(validators.Schema):
    uid = validators.Userid()


class BindPhonishAccountByUidHandler(InternalBrokerHandlerV2):
    """
    Связать два яндексовых аккаунта по токенам.
    """
    basic_form = _BindPhonishAccountByUidForm
    required_grants = ['bind-phonish-account-by-uid']

    def _process_request(self):
        portal_account = self._get_account_from_token()
        phonish_account = self._get_account_from_uid(self.form_values['uid'])

        if not phonish_account.is_phonish:
            logger.debug('Uid belongs not to phonish account')
            raise ProfileNotAllowedWebServiceError()
        phone_number = phonish_account.phones.default.number

        phone_on_portal_account = portal_account.phones.by_number(phone_number)
        if phone_on_portal_account is None:
            logger.debug('Portal account does not have phonish phone')
            raise ProfileNotAllowedWebServiceError()

        portal_bound_time = phone_on_portal_account.bound
        phonish_confirmed_time = phonish_account.phones.by_number(phone_number).confirmed
        portal_confirmed_time = phone_on_portal_account.confirmed

        if not portal_bound_time:
            logger.debug('Bound time undefined for portal account')
            raise ProfileNotAllowedWebServiceError()

        if not (portal_bound_time <= phonish_confirmed_time <= portal_confirmed_time):
            logger.debug('Time condition for binding is not fulfilled')
            raise ProfileNotAllowedWebServiceError()

        userinfo = {
            'userid': self.form_values['uid'],
            'provider': {
                'code': Yandex.code,
            },
        }
        try:
            profile_id = create_profile(
                mysql_read=get_slave_engine(),
                mysql_write=get_master_engine(),
                uid=portal_account.uid,
                profile_info=userinfo,
                token=None,
                timestamp=now.f(),
                yandexuid=None,
                blackbox=Blackbox(get_http_pool_manager()),
            )
            log_bind_profile(
                task_id=None,
                uid=portal_account.uid,
                profile_id=profile_id,
                profile_created=True,
            )
        except BaseProfileNotAllowedProfileCreationError:
            logger.debug('Profile not allowed')
            raise ProfileNotAllowedWebServiceError()

        self._statbox.log(action='update_account_yandex_bindings', uid=portal_account.uid)
        self._binding_log.log_event(
            BindingCreatedStatboxEvent(
                master_uid=portal_account.uid,
                slave_userid=userinfo['userid'],
                provider_code=Yandex.code,
            ),
        )
        invalidate_billing_cache(portal_account.uid)

    @cached_property
    def _statbox(self):
        return StatboxLogger(
            consumer=self._consumer,
            ip=self._user_ip,
        )

    @cached_property
    def _binding_log(self):
        return BindingLogger()
