# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

import flask
from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.broker.misc import invalidate_billing_cache
from passport.backend.social.broker.statbox import LongStatboxLogger
from passport.backend.social.common import validators
from passport.backend.social.common.builders.passport import (
    BasePassportError,
    Passport,
    PassportAccountDisabledError,
    PassportAccountNotFoundError,
    PassportPhoneOperationExistsError,
    PassportTemporaryError,
)
from passport.backend.social.common.misc import is_account_like_portalish
from passport.backend.social.common.multiprocessing import execute_multiple_methods
from passport.backend.social.common.profile import find_profiles_by_userid
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_logging import (
    BindingCreatedStatboxEvent,
    BindingLogger,
)
from passport.backend.social.common.token.domain import Token
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.validators import DeviceInfoForm


logger = logging.getLogger(__name__)


class _BindYandexAccountByTokenForm(validators.Schema):
    token = validators.Token()
    client_id = validators.ClientId()


class BindYandexAccountByTokenHandler(InternalBrokerHandlerV2):
    """
    Связать два яндексовых аккаунта по токенам.
    """
    basic_form = _BindYandexAccountByTokenForm
    required_grants = ['bind-yandex-account-by-token']

    def _process_request(self):
        master_account = self._get_account_from_token()

        app = self._get_application_from_client_id(
            provider_code=Yandex.code,
            client_id=self.form_values['client_id'],
        )
        client_token = Token(
            application_id=app.identifier,
            value=self.form_values['token'],
        )

        server_token, _ = self._sanitize_client_token(app, client_token)
        slave_userinfo = self._get_social_userinfo(app, server_token)
        slave_account = self._get_account_from_uid(slave_userinfo['userid'])
        logger.debug(
            'Bind Yandex accounts: master=%s, slave=%s' % (
                master_account.uid,
                slave_userinfo['userid'],
            )
        )

        @flask.copy_current_request_context
        def bind():
            self._bind_by_social_userinfo(
                master_account,
                slave_userinfo,
                # Токен не сохраняем, т.к. не проверяем его настоящий client_id.
                token=None,
            )
            self._binding_log.log_event(
                BindingCreatedStatboxEvent(
                    master_uid=master_account.uid,
                    slave_userid=slave_userinfo['userid'],
                    provider_code=Yandex.code,
                ),
            )
            self._statbox.log(action='update_account_yandex_bindings', uid=master_account.uid)

            invalidate_billing_cache(master_account.uid)

        @flask.copy_current_request_context
        def collect():
            return self._collect_statistics(master_account, slave_account)

        # Очень не хочется отнимать время запроса сбором статистики, поэтому
        # собираем её по возможности паралелльно.
        _, stats = execute_multiple_methods([bind, collect])

        # Важно привязывать телефон после сбора статистики, потому что по
        # другому мы не узнаём что фонишный телефон не был привязан к
        # портальному аккаунты до связывания.
        if is_account_like_portalish(master_account) and slave_account.is_phonish:
            self._bind_phone_from_phonish_to_portal(master_account.uid, slave_account.uid)

        if stats is not None and slave_account.is_phonish:
            stats = dict(stats, action='create_phonish_account_binding')
            self._statbox.log(**stats)

    @cached_property
    def _statbox(self):
        return LongStatboxLogger(
            consumer=self._consumer,
            ip=self._user_ip,
        )

    @cached_property
    def _binding_log(self):
        return BindingLogger()

    def _bind_phone_from_phonish_to_portal(self, master_uid, slave_uid):
        passport = Passport(get_http_pool_manager())
        try:
            passport.bind_phone_from_phonish_to_portal(
                portal_uid=master_uid,
                phonish_uid=slave_uid,
                user_ip=self._user_ip,
            )
        except (
            PassportPhoneOperationExistsError,
            PassportAccountNotFoundError,
            PassportAccountDisabledError,
            PassportTemporaryError,
        ) as e:
            # Т.к. транзакции здесь нет, то и сообщать об отказе, когда
            # основная работа уже сделана, мало пользы. Поэтому делим отказы на
            # 2 группы — те что могут произойти в результате гонки или состония
            # аккаунта, и те что с моей т.з. не должны возникать.
            logger.debug('Unable to bind phone from phonish to portal: %s' % e)
            self.response_values.update({'failed_to_bind_phone': True})
        except BasePassportError as e:
            logger.error('Unable to bind phone from phonish to portal: %s' % e)
            self.response_values.update({'failed_to_bind_phone': True})

    def _collect_statistics(self, master, slave):
        if not slave.is_phonish:
            return

        try:
            retval = dict(
                portal_uid=master.uid,
                phonish_uid=slave.uid,
            )

            if self._user_ip:
                retval.update(user_ip=str(self._user_ip))

            phonish_profiles = find_profiles_by_userid(
                str(slave.uid),
                Yandex.id,
                force_all=True,
            )
            master_uids = set(p.uid for p in phonish_profiles)
            master_uids.discard(master.uid)
            retval.update(master_uids=','.join(map(str, master_uids)))

            phone_bindings = self._blackbox.phone_bindings(
                need_current=True,
                need_history=False,
                need_unbound=False,
                phone_numbers=[slave.phones.default.number.e164],
            )
            phone_holder_uids = set(b['uid'] for b in phone_bindings)
            phone_holder_uids.discard(slave.uid)
            retval.update(phone_holder_uids=','.join(map(str, phone_holder_uids)))

            retval.update(is_phone_new_to_portal=master.uid not in phone_holder_uids)

            device_info = self._process_form(DeviceInfoForm())
            for key in device_info:
                # Добавим префикс к информации об устройстве, чтобы уменьшить
                # вероятность коллизии имён.
                retval.update({'device_info_' + key: device_info[key]})
        except Exception:
            logger.debug('Failed to collect statistics', exc_info=True)
            return
        return retval
