# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.social.broker.handlers.base import InternalBrokerHandlerV2
from passport.backend.social.common import validators
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.profile import (
    BaseProfileCreationError,
    ProfileCreator,
)
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.domain import Token


class _DoesBindingExistByTokenForm(validators.Schema):
    token = validators.Token()
    provider = validators.ProviderCode()
    client_id = validators.ClientId()


class DoesBindingExistByToken(InternalBrokerHandlerV2):
    """
    Сообщает о существовании профиля.

    Суффикс ByToken в имени класса относится к яндексовому аккаунту, а не к
    аккаунту в социальной сети.
    """
    required_grants = ['does-binding-exist-by-token']
    basic_form = _DoesBindingExistByTokenForm()

    def _process_request(self):
        account = self._get_account_from_token()

        app = self._get_application_from_client_id(
            provider_code=self.form_values['provider'],
            client_id=self.form_values['client_id'],
        )
        client_token = Token(
            application_id=app.identifier,
            value=self.form_values['token'],
        )

        server_token, _ = self._sanitize_client_token(app, client_token)
        userinfo = self._get_social_userinfo(app, server_token)

        self.response_values.update({
            'is_account_bound': self._does_binding_exist(account, userinfo),
            'is_possible': self._is_profile_possible(account, userinfo),
            'offer_delays': social_config.account_binding_offer_delays,
        })

    def _is_profile_possible(self, account, userinfo):
        profile_creator = ProfileCreator(
            mysql_read=get_slave_engine(),
            mysql_write=get_master_engine(),
            uid=account.uid,
            social_userinfo=userinfo,
            token=None,
            timestamp=now(),
            yandexuid=None,
            blackbox=self._blackbox,
        )
        try:
            profile_creator.check_profile_possible()
            return True
        except BaseProfileCreationError:
            return False
