# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from copy import copy
import json
import logging
import sys

from furl import furl
from passport.backend.social.broker.exceptions import RetpathInvalidError
from passport.backend.social.broker.handlers.base import Handler
from passport.backend.social.broker.misc import (
    check_url,
    RE_INVALID_HOSTS,
    UrlGrammar,
)
from passport.backend.social.common import validators
from passport.backend.social.common.exception import NotFound
from passport.backend.social.common.misc import (
    GraphiteMessageType,
    write_graph_log_message,
)
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.task import (
    load_task_from_redis,
    TaskId,
)
from passport.backend.social.common.useragent import Url
from passport.backend.utils.string import smart_bytes


VALID_CALLBACK_GRAMMARS = [
    UrlGrammar(
        """
        domain = subdomain '.yandex.' yandex_tld
        subdomain = 'social-test' |
            'social-rc' |
            'social' |
            domain_bit '.social-dev'
        path = web_callback | authz_in_app_callback | authz_in_web_callback
        web_callback = '/broker2/' callback_tail
        authz_in_app_callback = '/broker2/authz_in_app/' callback_tail
        authz_in_web_callback = '/broker2/authz_in_web/' callback_tail
        callback_tail = path_bit '/callback' '/'?
        """
    ),
]

logger = logging.getLogger('social.broker.handlers')


class RedirectHandler(Handler):
    def get(self):
        self.check_grant('redirect')

        write_graph_log_message(GraphiteMessageType.other, self.method_name)

        args = copy(self.request.args)

        state = args.pop('url', None) or args.pop('state', None)
        url = self.get_url_from_state(state)
        url = furl(url)

        # Значения словаря декодированы, но в furl нужно положить
        # закодированные значения.
        args = dict([(name, args[name].encode('utf-8')) for name in args])
        url.args.update(args)

        self.response.data = json.dumps({'location': url.url})
        return self.response

    @property
    def consumer(self):
        return self._get_consumer_from_headers()

    def get_url_from_state(self, state):
        if not state:
            raise RetpathInvalidError('Empty retpath')

        try:
            url = CallbackUrl.from_str(state)
            url.validate()
            return smart_bytes(url)
        except RetpathInvalidError:
            old_exc_type, old_exc_value, old_exc_traceback = sys.exc_info()

        try:
            task_id = validators.TaskId.to_python(state)
        except validators.Invalid:
            raise old_exc_type, old_exc_value, old_exc_traceback

        try:
            environment_id = TaskId(task_id).environment_id
        except ValueError:
            raise old_exc_type, old_exc_value, old_exc_traceback

        try:
            if environment_id != social_config.environment_id:
                url = self.build_redirect_url_to_other_environment(environment_id, state)
            else:
                url = self.load_url_from_task(task_id)
            return smart_bytes(url)
        except NotFound:
            raise old_exc_type, old_exc_value, old_exc_traceback

    def build_redirect_url_to_other_environment(self, environment_id, state):
        env = social_config.environment_from_id.get(environment_id)
        if not env:
            raise NotFound()
        return Url(social_config.social_broker_redirect_url_from_enviroment[env], params=dict(state=state))

    def load_url_from_task(self, task_id):
        task = load_task_from_redis(get_redis(), task_id)
        if not(task and task.callback_url):
            raise NotFound()

        return task.callback_url


class CallbackUrl(Url):
    @classmethod
    def from_str(cls, s):
        if type(s) is CallbackUrl:
            return s
        return cls(s)

    def validate(self):
        check_url(VALID_CALLBACK_GRAMMARS, RE_INVALID_HOSTS, self.paramless)
