# -*- coding: utf-8 -*-
"""
Пакет ручек, которыми нужно получать авторизацию для мобильного приложения или
другого клиента, которому известен X-Токен от аккаунта.
"""

from __future__ import unicode_literals

import logging

from passport.backend.social.broker.communicators.communicator import AuthorizeOptions
from passport.backend.social.broker.exceptions import PkceVerifierInvalidError
from passport.backend.social.broker.handlers.base import (
    CallbackHandler as _BaseCallbackHandler,
    Handler,
)
from passport.backend.social.broker.handlers.token import task_state
from passport.backend.social.broker.handlers.token.base import AUTHZ_IN_APP_COOKIE_PATH
from passport.backend.social.broker.misc import generate_retpath
from passport.backend.social.broker.statbox import to_statbox
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.pkce import (
    check_pkce,
    PkceInvalidCodeVerifierError,
)
from passport.backend.social.common.refresh_token.utils import save_refresh_token
from passport.backend.social.common.token.utils import (
    find_token_by_value_for_account,
    save_token,
)


logger = logging.getLogger('social.broker.handlers')


def build_callback_url(task_id, frontend_url):
    return '%sauthz_in_app/%s/callback' % (frontend_url, task_id)


def build_continue_url(task_id, frontend_url):
    return '%sauthz_in_app/%s/continue' % (frontend_url, task_id)


def get_better_url_from_app_token_handler_state(handler):
    """
    Подыскивает ручку, которая лучше подходит для состояния в котором находится
    вызвавший ручку пользователь.
    """
    if type(handler) is not AuthzInAppForTokenContinueHandler and handler.task.state == task_state.TOKEN_R_TO_CONT:
        return handler.build_continue_url()


class AuthzInAppForTokenStartHandler(Handler):
    COOKIE_PATH = AUTHZ_IN_APP_COOKIE_PATH

    def get(self):
        self.check_grant('authz-in-app')
        self.process_args()
        self.log_that_request_is_ok()
        self.create_communicator()

        self.init_new_task()

        self.fill_task_with_processed_args()

        task_cookie = self.save_task_to_cookie()

        authorization_url = self.build_authorization_url(self.processed_args['yandex_auth_code'])

        self.response.data = self.compose_json_response(
            dict(
                location=authorization_url,
                cookies=[task_cookie],
            ),
        )

        logger.info('Redirecting to "%s"' % authorization_url)
        return self.response

    def process_args(self):
        processor = self.build_request_processor()

        processor.process_consumer()
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        processor.process_retpath()
        processor.fix_morda_retpath()
        processor.process_place()
        processor.process_passthrough_errors()
        self.processed_args.update(processor.processed_args)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_display()
        processor.process_application_name(provider_is_required=False)
        processor.process_place()
        processor.process_user_ip()
        processor.process_ui_language()

        processor.process_yandex_auth_code()
        processor.process_pkce()
        processor.process_user_param()

        self.processed_args.update(processor.processed_args)

    def fill_task_with_processed_args(self):
        # Сейчас таску невозможно сохранить, пока start_args пустой, так что
        # положим туда что-нибудь.
        self.task.start_args = dict(foo='bar')
        self.task.consumer = self.processed_args['consumer']
        self.task.provider = self.processed_args['provider']
        self.task.application = self.processed_args['application']
        self.task.retpath = self.processed_args['retpath']
        self.task.place = self.processed_args['place']
        self.task.passthrough_errors = self.processed_args['passthrough_errors']
        self.task.code_challenge = self.processed_args['code_challenge']
        self.task.code_challenge_method = self.processed_args['code_challenge_method']
        self.task.state = task_state.TOKEN_R_TO_SOC
        self.task.user_param = self.processed_args['user_param']

    def log_that_request_is_ok(self):
        super(AuthzInAppForTokenStartHandler, self).log_that_request_is_ok()
        app = self.processed_args['application']
        provider_info = self.processed_args['provider']
        record = dict(
            mode='authz_in_app',
            action='start.submit',
            task_id=self.task_id,
            provider=None if not provider_info else provider_info['name'],
            application=app.name,
            retpath=self.processed_args['retpath'],
            place=self.processed_args['place'],
            ip=self.processed_args['user_ip'],
            consumer=self.processed_args['consumer'],
        )
        to_statbox(record)

    def build_authorization_url(self, yandex_auth_code):
        callback_url = build_callback_url(self.task_id, self.processed_args['frontend_url'])
        return self.communicator.get_authorize_url(AuthorizeOptions(
            callback_url,
            scope=self.communicator.get_scope(),
            user_param=self.processed_args.get('user_param'),
            yandex_auth_code=yandex_auth_code,
        ))

    @property
    def consumer(self):
        return self._get_consumer_from_headers()


class AuthzInAppForTokenCallbackHandler(_BaseCallbackHandler):
    EXPECTED_STATE = task_state.TOKEN_R_TO_SOC
    NEXT_STATE = task_state.TOKEN_R_TO_CONT
    COOKIE_PATH = AUTHZ_IN_APP_COOKIE_PATH

    def get(self, *args, **kwargs):
        self.check_grant('authz-in-app')
        return super(AuthzInAppForTokenCallbackHandler, self).get(*args, **kwargs)

    def build_continue_url(self):
        return build_continue_url(self.task_id, self.processed_args['frontend_url'])

    def process_callback_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        self.processed_args['retpath'] = self.task.retpath
        self.processed_args['place'] = self.task.place

        processor = self.build_request_processor()

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        processor.process_ui_language()

        self.processed_args.update(processor.processed_args)

        self.processed_args['provider'] = self.task.provider

        self.processed_args['application'] = self.task.application
        request_ctx.application = self.task.application

        self.processed_args['display'] = None
        self.processed_args['consumer'] = self.task.consumer

    @property
    def consumer(self):
        return self._get_consumer_from_headers()

    def find_better_handler_url_for_state(self):
        return get_better_url_from_app_token_handler_state(self)


class AuthzInAppForTokenContinueHandler(Handler):
    COOKIE_PATH = AUTHZ_IN_APP_COOKIE_PATH

    def get(self, task_id):
        self.check_grant('authz-in-app')

        self.task_id = task_id

        self.load_task_from_cookie()
        if self.task.in_redis:
            self.update_task_from_redis()

        self.process_args()
        self.assert_state([task_state.TOKEN_R_TO_CONT])
        self.log_that_request_is_ok()

        self.create_communicator()
        if not self.task.access_token:
            self.get_access_token(self.processed_args['frontend_url'], scopes=None)
        self.task.finished = now.f()
        self.save_task_to_redis()

        self.compose_response()
        return self.response

    def process_args(self):
        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        self.processed_args['retpath'] = self.task.retpath
        self.processed_args['place'] = self.task.place

        processor = self.build_request_processor()

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        processor.process_user_ip()
        processor.process_ui_language()

        self.processed_args.update(processor.processed_args)

        self.processed_args['provider'] = self.task.provider

        self.processed_args['application'] = self.task.application
        request_ctx.application = self.task.application

        self.processed_args['display'] = None
        self.processed_args['consumer'] = self.task.consumer

    def compose_response(self):
        task_cookie = self.save_task_to_cookie()

        location = generate_retpath(
            self.processed_args['retpath'],
            self.processed_args['place'],
            'ok',
            task_id=self.task_id,
        )

        self.response.data = self.compose_json_response(
            dict(
                location=location,
                cookies=[task_cookie],
            ),
        )

    def create_redirect_url_for_communicator(self, task_id, frontend_url):
        return build_callback_url(task_id, frontend_url)

    @property
    def consumer(self):
        return self._get_consumer_from_headers()


class AuthzInAppForTokenEntrustToAccountHandler(Handler):
    def get(self):
        self.check_grant('authz-in-app-entrust-to-account')

        self.process_args()
        self.log_that_request_is_ok()

        self.get_authorization(
            frontend_hostname=None,
            session_id=None,
            oauth_token=self.processed_args['oauth_token'],
            require_auth=True,
        )

        self.check_pkce(
            self.processed_args['code_verifier'],
            self.task.code_challenge,
            self.task.code_challenge_method,
        )

        app_name = self.processed_args['application'].name
        logger.debug('Bind token of application %s to account %s' % (app_name, self.task.uid))
        self.entrust_token(
            self.task.uid,
            self.task.get_token(),
            self.task.get_refresh_token(),
        )

        return self.compose_response()

    def process_args(self):
        processor = self.build_request_processor()

        self.process_yandex_token_args(processor)
        processor.process_task_id()
        processor.process_pkce_verifier()

        self.processed_args.update(processor.processed_args)

        self.task_id = self.processed_args['task_id']
        self.update_task_from_redis()

        self.processed_args.update(
            dict(
                application=self.task.application,
            ),
        )
        request_ctx.application = self.task.application

    def check_pkce(self, verifier, challenge, challenge_method):
        try:
            check_pkce(verifier, challenge, challenge_method)
        except PkceInvalidCodeVerifierError:
            raise PkceVerifierInvalidError()

    def entrust_token(self, uid, token, refresh_token):
        token.uid = uid
        timestamp = now()
        token.created = timestamp
        token.verified = timestamp
        token.confirmed = timestamp

        new_token = token

        old_token = find_token_by_value_for_account(uid, new_token.application_id, new_token.value, db=get_slave_engine())
        if old_token:
            old_token.update_with_reissued_token(new_token)
            token = old_token
        save_token(token, get_master_engine())

        if refresh_token:
            refresh_token.token_id = token.token_id
            save_refresh_token(refresh_token, get_master_engine())

    def compose_response(self):
        self.response.data = self.compose_json_response(dict(status='ok'))
        return self.response

    @property
    def consumer(self):
        return self._get_consumer_from_headers()
