# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.broker import exceptions as broker_exceptions
from passport.backend.social.broker.communicators.communicator import AuthorizeOptions
from passport.backend.social.broker.failure_diagnostics import (
    DiagnosticCouncil,
    DiagnosticInternalBrokerHandlerV1Mixin,
    DiagnosticManager,
    FailureDiagnostics,
    load_failure_diagnostics,
    StopDiagnosticsException,
)
from passport.backend.social.broker.handlers.args_processor import ArgsProcessor
from passport.backend.social.broker.handlers.base import (
    CallbackHandler,
    Handler,
    InternalBrokerHandlerV2,
    STATUS_REDIRECT,
)
from passport.backend.social.broker.handlers.token import task_state
from passport.backend.social.broker.handlers.token.base import AUTHZ_IN_WEB_COOKIE_PATH
from passport.backend.social.broker.misc import generate_retpath
from passport.backend.social.broker.statbox import to_statbox
from passport.backend.social.common import (
    exception as proxylib_exceptions,
    oauth2,
    useragent,
    validators,
)
from passport.backend.social.common.application import ApplicationGroupId
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.misc import urlencode
from passport.backend.social.common.redis_client import RedisError
from passport.backend.social.common.refresh_token.utils import save_refresh_token
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.utils import (
    find_all_tokens_for_account,
    find_token_by_value_for_account,
    save_token,
)
from passport.backend.social.common.useragent import Url
from passport.backend.social.common.web_service import (
    ApplicationUnknownWebServiceError,
    DatabaseFailedWebServiceError,
)
from passport.backend.social.proxylib import get_proxy
from passport.backend.utils.common import noneless_dict


logger = logging.getLogger('social.broker.handlers')

STATUS_FORM_BIND_AGREEMENT = 'form_bind_agreement'
STATUS_FORM_TRY_AGAIN = 'form_try_again'


def build_callback_url(task_id, frontend_url):
    return '%sauthz_in_web/%s/callback' % (frontend_url, task_id)


def build_collect_diagnostics_url(task_id, frontend_url):
    return '%sauthz_in_web/%s/collect_diagnostics' % (frontend_url, task_id)


def build_bind_url(task_id, frontend_url):
    return '%sauthz_in_web/%s/bind' % (frontend_url, task_id)


def build_start_url(frontend_url, **kwargs):
    url = Url(
        '%sauthz_in_web/start' % frontend_url,
        params=kwargs,
    )
    return str(url)


def get_better_url_from_web_token_handler_state(handler):
    """
    Подыскивает ручку, которая лучше подходит для состояния в котором находится
    вызвавший ручку пользователь.
    """
    if (
        type(handler) is not AuthzInWebForTokenBindSubmitHandler and
        type(handler) is not AuthzInWebForTokenBindCommitHandler and
        (
            handler.task.state == task_state.AUTHZ_IN_WEB_GOT_AUTH_CODE or
            handler.task.state == task_state.AUTHZ_IN_WEB_GOT_TOKEN
        )
    ):
        return handler.build_bind_url()


class _AuthzInWebForTokenMixin(object):
    COOKIE_PATH = AUTHZ_IN_WEB_COOKIE_PATH

    @property
    def consumer(self):
        return self._get_consumer_from_headers()

    def process_args_from_task(self):
        self.processed_args['consumer'] = self.task.consumer
        self.processed_args['provider'] = self.task.provider
        self.processed_args['application'] = self.task.application
        self.processed_args['retpath'] = self.task.retpath
        self.processed_args['place'] = self.task.place

    def entrust_token(self, uid, token, refresh_token):
        token.uid = uid
        timestamp = now()
        token.created = timestamp
        token.verified = timestamp
        token.confirmed = timestamp

        new_token = token

        old_token = find_token_by_value_for_account(
            uid,
            new_token.application_id,
            new_token.value,
            db=get_slave_engine(),
        )
        if old_token:
            old_token.update_with_reissued_token(new_token)
            token = old_token
        save_token(token, get_master_engine())

        if refresh_token:
            refresh_token.token_id = token.token_id
            save_refresh_token(refresh_token, get_master_engine())

    def create_redirect_url_for_communicator(self, task_id, frontend_url):
        return build_callback_url(task_id, frontend_url)

    def compose_redirect_to_success_retpath_response(self, keep_task_cookie=True):
        if keep_task_cookie:
            task_cookie = self.save_task_to_cookie()
        else:
            task_cookie = self.burn_task_cookie()
        success_url = generate_retpath(
            self.processed_args['retpath'],
            self.processed_args['place'],
            'ok',
        )
        return self.compose_json_response(
            dict(
                cookies=[task_cookie],
                status=STATUS_REDIRECT,
                location=success_url,
            ),
        )

    def build_retpath_fail_url(self):
        return generate_retpath(
            self.processed_args['retpath'],
            self.processed_args['place'],
            'error',
        )

    def build_authorization_url(self):
        callback_url = build_callback_url(self.task_id, self.processed_args['frontend_url'])
        return self.communicator.get_authorize_url(AuthorizeOptions(
            callback_url,
            scope=self.communicator.get_scope(),
            user_param=self.processed_args.get('user_param'),
        ))

    def compose_redirect_to_fail_retpath_response(self):
        """
        Ответ с редиректом на fail_retpath

        Отличается от выброса исключения тем, что пользователю не показывается
        окно с сообщением, что произошёл отказ.
        """
        task_cookie = self.burn_task_cookie()
        fail_url = self.build_retpath_fail_url()
        return self.compose_json_response(
            dict(
                status=STATUS_REDIRECT,
                location=fail_url,
                cookies=[task_cookie],
            ),
        )


class _AuthzInWebForTokenDiagnosticHandlerMixin(DiagnosticInternalBrokerHandlerV1Mixin):
    def build_diagnostic_manager(self):
        return DiagnosticManager(_AuthzinInWebForTokenDiagnosticCouncil())

    def compose_diagnostic_response(self):
        return self.compose_redirect_to_fail_retpath_response()


class AuthzInWebForTokenStartHandler(_AuthzInWebForTokenMixin, Handler):
    def get(self):
        self.check_grant('authz-in-web-for-token')
        self.process_args()
        self.log_that_request_is_ok()
        self.create_communicator()

        self.init_new_task()
        self.fill_task_with_processed_args()

        try:
            self.get_authorization(
                frontend_hostname=self.processed_args['hostname'],
                session_id=self.processed_args['Session_id'],
                oauth_token=None,
                require_auth=True,
            )
        except broker_exceptions.AuthorizationRequiredError:
            self.response.data = self.compose_redirect_to_passport()
            return self.response

        task_cookie = self.save_task_to_cookie()

        authorization_url = self.build_authorization_url()

        self.response.data = self.compose_json_response(
            dict(
                status=STATUS_REDIRECT,
                location=authorization_url,
                cookies=[task_cookie],
            ),
        )

        logger.info('Redirecting to "%s"' % authorization_url)
        return self.response

    def process_args(self):
        processor = ArgsProcessor(self.request.form, self.request.form)

        processor.process_consumer()

        # retpath и place нужно парсить одним из первых, т.к. туда фронт будет
        # направлять веб-клиент пользователя в случаях отказа.
        processor.process_retpath()
        processor.process_place()
        self.processed_args.update(processor.processed_args)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_display()
        processor.process_application_name(provider_is_required=False)
        processor.process_session_id()
        processor.process_user_ip()
        processor.process_ui_language()
        processor.process_flags()
        processor.process_user_param()
        self.processed_args.update(processor.processed_args)

    def fill_task_with_processed_args(self):
        # Сейчас таску невозможно сохранить, пока start_args пустой, так что
        # положим туда что-нибудь.
        self.task.start_args = dict(foo='bar')

        self.task.application = self.processed_args['application']
        self.task.consumer = self.processed_args['consumer']
        self.task.place = self.processed_args['place']
        self.task.provider = self.processed_args['provider']
        self.task.retpath = self.processed_args['retpath']
        self.task.state = task_state.AUTHZ_IN_WEB_R_TO_SOC
        self.task.user_param = self.processed_args['user_param']

        if self.processed_args['flags'] and 'collect_diagnostics' in self.processed_args['flags']:
            self.task.collect_diagnostics = True

    def log_that_request_is_ok(self):
        super(AuthzInWebForTokenStartHandler, self).log_that_request_is_ok()
        app = self.processed_args['application']
        provider_info = self.processed_args['provider']
        record = dict(
            mode='authz_in_web',
            action='start.submit',
            task_id=self.task_id,
            provider=None if not provider_info else provider_info['name'],
            application=app.name,
            retpath=self.processed_args['retpath'],
            place=self.processed_args['place'],
            ip=self.processed_args['user_ip'],
            consumer=self.processed_args['consumer'],
        )
        to_statbox(record)

    def compose_redirect_to_passport(self):
        start_url = self.build_start_url()
        retpath_fail_url = self.build_retpath_fail_url()
        passport_url = self.build_passport_url(
            retpath=start_url,
            backpath=retpath_fail_url,
        )
        return self.compose_json_response(
            dict(
                status=STATUS_REDIRECT,
                location=passport_url,
            ),
        )

    def build_passport_url(self, retpath, backpath):
        passport_host = social_config.passport_frontend_host_without_tld + self.processed_args['tld']
        url = Url(
            'https://%s/auth' % passport_host,
            params={
                'from': 'social',
                'retpath': retpath,
                'backpath': backpath,
            },
        )
        return str(url)

    def build_start_url(self):
        return build_start_url(
            self.processed_args['frontend_url'],
            application_name=self.processed_args['application'].name,
            consumer=self.processed_args['consumer'],
            retpath=self.processed_args['retpath'],
            user_param=self.processed_args['user_param'],
        )


class AuthzInWebForTokenCallbackHandler(_AuthzInWebForTokenMixin, CallbackHandler):
    def get(self):
        self.check_grant('authz-in-web-for-token')

        self.process_args()

        self.task_id = self.processed_args['task_id']
        self.load_task_from_cookie()

        self.process_args_from_task()

        if self.find_better_handler_url_for_state():
            return self.compose_send_to_better_handler_for_state_response()

        self.assert_state([task_state.AUTHZ_IN_WEB_R_TO_SOC])
        self.log_that_request_is_ok()

        self.create_communicator()

        if self.task.collect_diagnostics:
            self.diagnostic_manager.enable(self.task.application.name, append=False)

        try:
            exchange = self.get_exchange_value_from_callback()
        except broker_exceptions.UserDeniedError as e:
            self.response.data = self.compose_form_try_again_response()
            if self.task.collect_diagnostics:
                raise StopDiagnosticsException(e, on_fail_return=self.response)
            return self.response

        self.task.state = task_state.AUTHZ_IN_WEB_GOT_AUTH_CODE
        self.task.exchange = exchange

        if self.task.collect_diagnostics:
            self.response.data = self.compose_redirect_to_collect_diagnostics_response()
        else:
            self.response.data = self.compose_redirect_to_bind_response()

        return self.response

    def process_args(self):
        processor = ArgsProcessor(self.request.form, self.request.form)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_task_id()
        processor.process_query()
        processor.process_session_id()
        processor.process_user_ip()
        processor.process_display()
        processor.process_ui_language()
        self.processed_args.update(processor.processed_args)

    def compose_form_try_again_response(self):
        app = self.task.application
        cancel_url = self.build_retpath_fail_url()
        retry_url = self.build_authorization_url()
        return self.compose_json_response(
            dict(
                status=STATUS_FORM_TRY_AGAIN,
                application_display_name=app.get_display_name_or_default(),
                cancel_url=cancel_url,
                retry_url=retry_url,
            ),
        )

    def compose_redirect_to_bind_response(self):
        task_cookie = self.save_task_to_cookie()
        return self.compose_json_response(
            dict(
                status=STATUS_REDIRECT,
                location=self.build_bind_url(),
                cookies=[task_cookie],
            ),
        )

    def build_bind_url(self):
        return build_bind_url(self.task_id, self.processed_args['frontend_url'])

    def compose_redirect_to_collect_diagnostics_response(self):
        task_cookie = self.save_task_to_cookie()
        collect_diagnostics_url = build_collect_diagnostics_url(self.task_id, self.processed_args['frontend_url'])
        return self.compose_json_response(
            dict(
                status=STATUS_REDIRECT,
                location=collect_diagnostics_url,
                cookies=[task_cookie],
            ),
        )

    def get_callback_query(self):
        return self.processed_args['query']

    def find_better_handler_url_for_state(self):
        return get_better_url_from_web_token_handler_state(self)


class AuthzInWebForTokenCallbackHandler(
    _AuthzInWebForTokenDiagnosticHandlerMixin,
    AuthzInWebForTokenCallbackHandler,
):
    pass


class AuthzInWebForTokenBindSubmitHandler(_AuthzInWebForTokenMixin, Handler):
    def get(self):
        self.check_grant('authz-in-web-for-token')

        self.process_args()

        self.task_id = self.processed_args['task_id']
        self.load_task_from_cookie()

        if (
            self.task.in_redis or
            self.task.state == task_state.AUTHZ_IN_WEB_GOT_TOKEN
        ):
            self.update_task_from_redis()

        self.process_args_from_task()

        self.assert_state([task_state.AUTHZ_IN_WEB_GOT_AUTH_CODE, task_state.AUTHZ_IN_WEB_GOT_TOKEN])

        self.log_that_request_is_ok()

        self.create_communicator()

        self.get_authorization(
            frontend_hostname=self.processed_args['hostname'],
            session_id=self.processed_args['Session_id'],
            oauth_token=None,
            require_auth=True,
        )

        if self.task.state == task_state.AUTHZ_IN_WEB_GOT_AUTH_CODE:
            self.get_access_token(self.processed_args['frontend_url'], scopes=None)
            self.task.state = task_state.AUTHZ_IN_WEB_GOT_TOKEN
            self.task.finished = now.f()
            self.save_task_to_redis()

        old_tokens = self.find_all_tokens_for_account(
            self.task.uid,
            application_ids=[self.task.application.identifier],
        )
        if old_tokens:
            # Т.к. пользователь уже привязывал токен от данного приложения,
            # значит он уже давал согасие на связывание, и в этот раз можно,
            # не спрашивая, привязать к его аккаунту новый токен.
            app_name = self.task.application.name
            logger.debug('Bind token of application %s to account %s' % (app_name, self.task.uid))
            self.entrust_token(
                self.task.uid,
                self.task.get_token(),
                self.task.get_refresh_token(),
            )

            self.response.data = self.compose_redirect_to_success_retpath_response()
            return self.response

        self.response.data = self.compose_form_bind_agreement_response()
        return self.response

    def process_args(self):
        processor = ArgsProcessor(self.request.form, self.request.form)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_task_id()
        processor.process_session_id()
        processor.process_user_ip()
        processor.process_display()
        processor.process_ui_language()
        self.processed_args.update(processor.processed_args)

    def compose_form_bind_agreement_response(self):
        task_cookie = self.save_task_to_cookie()
        app = self.task.application
        account_display_name = self.task.display_name
        cancel_url = self.build_retpath_fail_url()
        return self.compose_json_response(
            dict(
                status=STATUS_FORM_BIND_AGREEMENT,
                cookies=[task_cookie],
                application_display_name=app.get_display_name_or_default(),
                account_display_name=account_display_name,
                cancel_url=cancel_url,
            ),
        )

    def find_all_tokens_for_account(self, uid, application_ids):
        return find_all_tokens_for_account(uid, get_slave_engine(), application_ids)


class AuthzInWebForTokenBindCommitHandler(_AuthzInWebForTokenMixin, Handler):
    def get(self):
        self.check_grant('authz-in-web-for-token')

        self.process_args()

        self.task_id = self.processed_args['task_id']
        self.load_task_from_cookie()

        if (
            self.task.in_redis or
            self.task.state == task_state.AUTHZ_IN_WEB_GOT_TOKEN
        ):
            self.update_task_from_redis()

        self.process_args_from_task()

        self.assert_state([task_state.AUTHZ_IN_WEB_GOT_TOKEN])

        self.log_that_request_is_ok()

        self.get_authorization(
            frontend_hostname=self.processed_args['hostname'],
            session_id=self.processed_args['Session_id'],
            oauth_token=None,
            require_auth=True,
        )

        app_name = self.task.application.name
        logger.debug('Bind token of application %s to account %s' % (app_name, self.task.uid))
        self.entrust_token(
            self.task.uid,
            self.task.get_token(),
            self.task.get_refresh_token(),
        )

        self.response.data = self.compose_redirect_to_success_retpath_response()
        return self.response

    def process_args(self):
        processor = ArgsProcessor(self.request.form, self.request.form)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_task_id()
        processor.process_session_id()
        processor.process_user_ip()
        processor.process_display()
        processor.process_ui_language()
        self.processed_args.update(processor.processed_args)


class AuthzInWebForTokenCollectDiagnosticsHandler(_AuthzInWebForTokenMixin, Handler):
    def get(self):
        self.check_grant('authz-in-web-for-token')

        self.process_args()

        self.task_id = self.processed_args['task_id']
        self.load_task_from_cookie()

        self.process_args_from_task()

        self.diagnostic_manager.enable(self.task.application.name)

        self.assert_state([task_state.AUTHZ_IN_WEB_GOT_AUTH_CODE])
        self.create_communicator()
        self.get_authorization(
            frontend_hostname=self.processed_args['hostname'],
            session_id=self.processed_args['Session_id'],
            oauth_token=None,
            require_auth=True,
        )

        # Зафиксируем время до выполнения запросов в провайдера, чтобы не
        # приходилось учитывать эти издержки, когда будем вычислять TTL токена.
        timestamp = now.f()

        self.diagnostic_manager.context = 'get_access_token'
        self.get_access_token(self.processed_args['frontend_url'], scopes=None)

        if self.task.application.refresh_token_url:
            proxy = get_proxy(
                app=self.task.application,
                should_retry_on_invalid_token=True,
            )

            if self.task.access_token.get('refresh'):
                self.diagnostic_manager.context = 'refresh_valid_token'
                proxy.refresh_token(self.task.access_token['refresh'])

            self.diagnostic_manager.context = 'refresh_invalid_token'
            try:
                proxy.refresh_token('invalid')
            except proxylib_exceptions.InvalidTokenProxylibError:
                logger.debug('Refresh invalid token test passed')

        if (
            self.task.access_token.get('expires') and
            not self.task.application.refresh_token_url
        ):
            self.diagnostic_manager.build_manual_report(TokenExpirableButNoRefreshTokenUrlFailureDiagnostics())

        if (
            self.task.access_token.get('expires') and
            not self.task.access_token.get('refresh')
        ):
            self.diagnostic_manager.build_manual_report(TokenExpirableButNoRefreshTokenFailureDiagnostics())

        if self.task.access_token.get('expires') is not None:
            if self.task.access_token['expires'] - int(timestamp) < social_config.diagnostics_recommended_token_ttl:
                self.diagnostic_manager.build_manual_report(LowTokenExpiresInFailureDiagnostics())

        self.response.data = self.compose_redirect_to_success_retpath_response(keep_task_cookie=False)
        return self.response

    def process_args(self):
        processor = ArgsProcessor(self.request.form, self.request.form)

        processor.process_frontend_url()
        processor.process_hostname_and_tld(processor.processed_args['frontend_url'])
        self.processed_args.update(processor.processed_args)

        processor.process_task_id()
        processor.process_session_id()
        processor.process_user_ip()
        processor.process_display()
        processor.process_ui_language()
        self.processed_args.update(processor.processed_args)


class AuthzInWebForTokenCollectDiagnosticsHandler(
    _AuthzInWebForTokenDiagnosticHandlerMixin,
    AuthzInWebForTokenCollectDiagnosticsHandler,
):
    pass


class _AuthzInWebForTokenReadDiagnosticsForm(validators.Schema):
    application_name = validators.ApplicationName()


class AuthzInWebForTokenReadDiagnosticsHandler(InternalBrokerHandlerV2):
    required_grants = ['authz-in-web-for-token-read-diagnostics']
    basic_form = _AuthzInWebForTokenReadDiagnosticsForm()

    def _process_request(self):
        app = self._get_application_from_name(self.form_values['application_name'])

        if app.group_id != ApplicationGroupId.station:
            raise ApplicationUnknownWebServiceError()

        try:
            diagnostics = load_failure_diagnostics(app.name)
        except RedisError:
            raise DatabaseFailedWebServiceError()

        diagnostics = [d.to_response_dict() for d in diagnostics]
        self.response_values.update(diagnostics=diagnostics)


class _AuthzinInWebForTokenDiagnosticCouncil(DiagnosticCouncil):
    def get_experts(self):
        return [
            self.handle_zora_dns_failure,
            self.handle_zora_ssl_failure,
            self.handle_token_response_error,
            self.handle_useragent_request_timeout,
            self.handle_bad_http_status,
            self.handle_missing_authorization_code,
            self.handle_authorization_error_response,
            self.handle_token_response_too_long,
            self.handle_token_response_invalid_attribute,
            self.handle_token_response_not_json,
        ]

    def handle_zora_dns_failure(self):
        # Проблемы с DNS через Zora
        for exc in self.find_exceptions(useragent.ZoraError):
            if exc.error == useragent.ZORA_DNS_FAILURE:
                exc = self.find_first_exception(useragent.NetworkFailureUseragentError)
                if exc:
                    url = useragent.Url(exc.url)
                    hostname = url.hostname
                else:
                    hostname = None
                fd = dict(
                    type='dns_failed',
                    hostname=hostname,
                )
                return FailureDiagnostics(fd)

    def handle_zora_ssl_failure(self):
        # Проблемы с SSL через Zora
        for exc in self.find_exceptions(useragent.ZoraError):
            if exc.error == useragent.ZORA_SSL_CERT_ERROR:
                exc = self.find_first_exception(useragent.NetworkFailureUseragentError)
                if exc:
                    if isinstance(exc, useragent.NetworkFailureUseragentError):
                        url = useragent.Url(exc.url)
                        url = url.paramless
                else:
                    url = None
                fd = dict(
                    type='ssl_failed',
                    url=url,
                )
                return FailureDiagnostics(fd)

    def handle_useragent_request_timeout(self):
        exc = self.find_first_exception(useragent.TimeoutUseragentError)
        if exc:
            url = useragent.Url(exc.url)
            return FailureDiagnostics(
                dict(
                    type='useragent_request_timeout',
                    url=url.paramless,
                    duration=exc.duration,
                ),
            )

    def handle_bad_http_status(self):
        if self.find_first_exception(broker_exceptions.BadHttpStatusCommunicationFailedError):
            exc = self.find_first_exception(useragent.ServiceTemporaryUnavailableUseragentError)
            if exc:
                url = useragent.Url(exc.url)
                # Отбросим параметры, для того чтобы исключить попадание
                # в диагностику секретных.
                url = url.paramless

                http_status = exc.http_status
            else:
                url = None
                http_status = None

            fd = dict(
                type='bad_http_status',
                http_status=None if http_status is None else str(http_status),
                url=url,
            )
            return FailureDiagnostics(fd)

    def handle_missing_authorization_code(self):
        exc = self.find_first_exception(broker_exceptions.MissingExchangeValueCommunicationFailedError)
        if exc:
            query = exc.query
            if query is not None:
                query = urlencode(query)
            fd = dict(
                type='missing_authorization_code',
                query=query
            )
            return FailureDiagnostics(fd)

    def handle_authorization_error_response(self):
        exceptions = (
            broker_exceptions.AuthorizationErrorResponseCommunicationFailedError,
            broker_exceptions.UserDeniedError,
        )
        exc = self.find_first_exception(exceptions)
        if exc:
            query = exc.query
            if query is not None:
                query = urlencode(query)
            fd = dict(
                type='authorization_error_response',
                query=query
            )
            return FailureDiagnostics(fd)

    def handle_token_response_too_long(self):
        exc = self.find_first_exception(proxylib_exceptions.TooLongUnexpectedResponseProxylibError)
        if exc:
            max_response_length = exc.max_response_length
            response_length = exc.response_length

            exc = self.find_first_exception(useragent.ServiceTemporaryUnavailableUseragentError)
            if exc:
                url = useragent.Url(exc.url)
                # Отбросим параметры, для того чтобы исключить попадание
                # в диагностику секретных.
                url = url.paramless

                http_status = exc.http_status
            else:
                url = None
                http_status = None

            fd = dict(
                type='token_response_too_long',
                http_status=None if http_status is None else str(http_status),
                max_response_length=max_response_length,
                response_length=response_length,
                url=url,
            )
            return FailureDiagnostics(fd)

    def handle_token_response_invalid_attribute(self):
        for exc in self.find_exceptions(proxylib_exceptions.InvalidResponseAttributeProxylibError):
            if exc.attr_name in {'access_token', 'expires_in', 'refresh_token'}:
                type_name = 'token_response_invalid_' + exc.attr_name
                max_length = exc.max_length
                min_length = exc.min_length

                fd = dict(
                    type=type_name,
                    max_length=max_length,
                    min_length=min_length,
                )
                fd = noneless_dict(fd)
                return FailureDiagnostics(fd)

    def handle_token_response_error(self):
        # В ответе код с отказом
        token_errors = (
            oauth2.token.InvalidClient,
            oauth2.token.InvalidGrant,
            oauth2.token.InvalidRequest,
            oauth2.token.InvalidScope,
            oauth2.token.UnauthorizedClient,
            oauth2.token.UnexpectedException,
            oauth2.token.UnsupportedGrantType,
        )
        exc = self.find_first_exception(token_errors)
        if exc:
            error = exc.error
            exc = self.find_first_exception(
                (
                    useragent.ParsingUseragentError,
                    useragent.ServiceTemporaryUnavailableUseragentError,
                ),
            )
            if exc:
                url = useragent.Url(exc.url)
                # Отбросим параметры, для того чтобы исключить попадание
                # в диагностику секретных.
                url = url.paramless
            else:
                url = None

            fd = dict(
                type='token_response_error',
                error=error,
                url=url,
            )
            return FailureDiagnostics(fd)

    def handle_token_response_not_json(self):
        if self.find_first_exception(proxylib_exceptions.ResponseDecodeProxylibError):
            exc = self.find_first_exception(useragent.ServiceTemporaryUnavailableUseragentError)
            if exc:
                url = useragent.Url(exc.url)
                # Отбросим параметры, для того чтобы исключить попадание
                # в диагностику секретных.
                url = url.paramless

                http_status = exc.http_status
            else:
                url = None
                http_status = None

            fd = dict(
                type='token_response_not_json',
                http_status=None if http_status is None else str(http_status),
                url=url,
            )
            return FailureDiagnostics(fd)


class TokenExpirableButNoRefreshTokenFailureDiagnostics(FailureDiagnostics):
    """
    Нет рефреш-токена, а время жизни токена ограничено

    Из-за этого не будет работать механизм автоматического обновления токена.
    А пользователю придётся перепроходить авторизацию, каждый раз когда
    протухает токен.
    """
    def __init__(self):
        diagnostics = dict(type='token_expirable_but_no_refresh_token')
        super(TokenExpirableButNoRefreshTokenFailureDiagnostics, self).__init__(diagnostics)


class TokenExpirableButNoRefreshTokenUrlFailureDiagnostics(FailureDiagnostics):
    """
    Для приложения не задан refresh token url, а время жизни токена ограничено

    Из-за этого не будет работать механизм автоматического обновления токена.
    А пользователю придётся перепроходить авторизацию, каждый раз когда
    протухает токен.
    """
    def __init__(self):
        diagnostics = dict(type='token_expirable_but_no_refresh_token_url')
        super(TokenExpirableButNoRefreshTokenUrlFailureDiagnostics, self).__init__(diagnostics)


class LowTokenExpiresInFailureDiagnostics(FailureDiagnostics):
    """
    Малое время жизни токена

    Плохо потому что создаётся бесполезная нагрузка на Социализм, да и на
    Oauth-сервер провайдера.
    """
    def __init__(self):
        diagnostics = dict(type='token_expires_in_low')
        super(LowTokenExpiresInFailureDiagnostics, self).__init__(diagnostics)
