# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from logging.config import dictConfig as logging_dict_config
import os
from os import path
import sys

from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.social_logging import (
    close_logging_handlers,
    ExceptionFormatter,
    SocialFormatter,
    WarningFormatter,
)


def logging_settings_init():
    logging.captureWarnings(True)
    conf = _get_logging_configuration()
    logging_dict_config(conf)


def logging_settings_deinit():
    conf = _get_logging_configuration()
    close_logging_handlers(conf)


def _get_logging_configuration():
    LOG_LEVEL = 'DEBUG'

    root_handlers = {'broker', 'broker.error', 'broker.exception', 'broker.warning'}
    subprocess_handlers = {'broker.subprocess.error'}
    if not os.environ.get('SYSTEM_WIDE_SOCIAL_BROKER'):
        root_handlers.add('console')
        subprocess_handlers.add('console')

    if os.environ.get('SYSTEM_WIDE_SOCIAL_BROKER'):
        LOG_PATH = '/var/log/yandex/socialism'
    else:
        LOG_PATH = '.'

    return {
        'version': 1,
        'disable_existing_loggers': False,
        'formatters': {
            'default': {
                '()': SocialFormatter,
                'context': request_ctx,
            },
            'exception': {
                '()': ExceptionFormatter,
                'context': request_ctx,
                'logtype': 'social-broker-exception-log',
            },
            'warning': {
                '()': WarningFormatter,
                'context': request_ctx,
                'logtype': 'social-broker-warning-log',
            },
            'simple': {
                'format': "%(message)s"
            },
        },
        'root': {
            'handlers': root_handlers,
            'level': LOG_LEVEL,
        },
        'handlers': {
            'console': {
                'class': 'logging.StreamHandler',
                'level': 'DEBUG',
                'formatter': 'default',
                'stream': sys.stdout,
            },
            'broker': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.debug.log'),
                'formatter': 'default',
                'level': LOG_LEVEL,
            },
            'broker.error': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.error.log'),
                'formatter': 'default',
                'level': 'ERROR',
            },
            'broker.exception': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-broker.exception.log'),
                'formatter': 'exception',
                'level': 'ERROR',
                'filters': ['exception_filter'],
            },
            'broker.warning': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-broker.warning.log'),
                'formatter': 'warning',
                'filters': ['warning_filter'],
            },
            'broker.subprocess.error': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.subprocess.error.log'),
                'formatter': 'default',
                'level': 'DEBUG',
            },
            'bindings_log': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-bindings.statbox.log'),
                'formatter': 'simple',
            },
            'statbox': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.statbox.log'),
                'formatter': 'simple',
            },
            'access': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.access.log'),
                'formatter': 'simple',
            },
            'graphite': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.graphite.log'),
                'formatter': 'simple',
            },
            'avatars': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': path.join(LOG_PATH, 'social-broker.avatars.log'),
                'formatter': 'simple',
            },
        },
        'filters': {
            'exception_filter': {
                '()': 'passport.backend.social.common.social_logging.ExceptionFilter',
            },
            'warning_filter': {
                '()': 'passport.backend.social.common.social_logging.LevelFilter',
                'levels': ['WARNING'],
            },
        },
        'loggers': {
            'bindings_log': {
                'handlers': ['bindings_log'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'statbox': {
                'handlers': ['statbox'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'subprocess.error': {
                'handlers': subprocess_handlers,
                'propagate': False,
            },
            'urllib3': {
                'level': logging.INFO,
            },
            'oauthlib': {
                'level': logging.INFO,
            },
            'access': {
                'handlers': ['access'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.database': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.useragent': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.redis': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'avatars': {
                'handlers': ['avatars'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'passport.backend.social.common._urllib3': {'level': 'WARNING'},
            'passport.backend.social.common.db.execute': {'level': 'WARNING'},
            'passport.backend.social.common.useragent': {'level': 'INFO'},

            # блок конфигурации для билдеров
            'passport.blackbox': {
                'handlers': root_handlers - {'broker.error'},
                'level': 'DEBUG',
                'propagate': False,
            },
            'passport.backend.social.common.builders.billing': {
                'handlers': root_handlers - {'broker.error'},
                'level': 'DEBUG',
                'propagate': False,
            },
        },
    }
