# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from base64 import urlsafe_b64encode
import logging
import re
from textwrap import dedent
import types
from urlparse import (
    urlparse,
    urlsplit,
    urlunparse,
)

from ometa.runtime import ParseError as ParsleyParseError
from parsley import makeGrammar as parsley_make_grammar
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.broker.exceptions import RetpathInvalidError
from passport.backend.social.common.builders.billing import invalidate_billing_cache as _invalidate_billing_cache
from passport.backend.social.common.grants import get_grants_config as _get_grants_config
from passport.backend.social.common.misc import (
    urlencode,
    urlparse_qs,
)
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import get_http_pool_manager


logger = logging.getLogger('broker.server')


def or_re_list(_list):
    return '(?:' + '|'.join(_list) + ')'


_VALID_TLDS = [
    'az', 'by', 'co.il', 'com', 'com.am', 'com.ge', 'com.tr', 'ee', 'eu', 'fi', 'fr',
    'kg', 'kz', 'lt', 'lv', 'md', 'net', 'pl', 'ru', 'tj', 'tm', 'ua', 'uz',
]

RE_TLD = re.compile(r'.*\.(%s)' % or_re_list(map(re.escape, _VALID_TLDS)))

RE_INVALID_HOSTS = re.compile(
    r'''
    # PASSP-14596 Не допускаем "%" в имени хоста, т.к. одни браузеры декодируют
    # его (ie11), а другие нет (firefox38.8).
    .*%.*
    .*\\.*
    ''',
    re.VERBOSE,
)

_BaseUrlGrammar = parsley_make_grammar(
    """
    yandex_tld = %s
    domain_bit = (letter | digit | '_' | '-')+
    path_bit = dotless_path_bit ('.' dotless_path_bit)*
    dotless_path_bit = (letter | digit | '_' | '-')+
    path = anything*
    """ % ' | '.join(sorted(["'%s'" % t for t in _VALID_TLDS], reverse=True)),
    dict(),
)


def UrlGrammar(source):
    return parsley_make_grammar(
        dedent(source),
        dict(),
        extends=_BaseUrlGrammar,
    )


def check_url(valid_grammars, invalid_hosts, url, extra_allowed_schemes=None):
    if not isinstance(url, basestring) or not url.strip():
        raise RetpathInvalidError('Empty retpath')

    url_components = urlsplit(url)

    basic_schemes = map(re.compile, ['^https$', '^http$'])
    valid_schemes = basic_schemes[:]
    if extra_allowed_schemes:
        valid_schemes += extra_allowed_schemes
    scheme = url_components.scheme.lower()
    if all(not valid_scheme.match(scheme) for valid_scheme in valid_schemes):
        raise RetpathInvalidError('Invalid scheme: "%s"' % url)

    if not url_components.hostname:
        raise RetpathInvalidError('Empty hostname: "%s"' % url)

    # Разрешаем любой URL, когда схема нестандартная
    if all(not basic_scheme.match(scheme) for basic_scheme in basic_schemes):
        return

    if invalid_hosts and invalid_hosts.match(url_components.netloc):
        raise RetpathInvalidError('Invalid netloc: "%s"' % url)

    for grammar in valid_grammars:
        try:
            grammar(url_components.netloc).domain()
            grammar(url_components.path).path()
            break
        except ParsleyParseError:
            pass
    else:
        raise RetpathInvalidError('Invalid netloc or path: "%s"' % url)


def check_retpath(url, extra_allowed_schemes=None):
    return check_url(_get_broker_retpath_grammars(), RE_INVALID_HOSTS, url, extra_allowed_schemes)


def generate_retpath(retpath, place, status, task_id=None, additional_args=None):
    logger.debug('Trying to build a retpath...')
    redirect_url = retpath
    redirector_parsed = list(urlparse(redirect_url))
    if (place or 'query') == 'query':
        item_id = 4
    else:
        item_id = 5
    qs = urlparse_qs(redirector_parsed[item_id])
    qs = dict((k, v if len(v) > 1 else v[0]) for k, v in qs.iteritems())
    qs['status'] = status
    if status == 'ok' and task_id:
        qs['task_id'] = task_id

    if additional_args:
        qs.update(additional_args)

    redirector_parsed[item_id] = urlencode(qs)
    retpath = urlunparse(redirector_parsed)
    logger.debug('retpath=' + unicode(retpath))
    return retpath


def smart_str(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Returns a bytestring version of 's', encoded as specified in 'encoding'.

    If strings_only is True, don't convert (some) non-string-like objects.
    """
    if strings_only and isinstance(s, (types.NoneType, int)):
        return s
    elif not isinstance(s, basestring):
        try:
            return str(s)
        except UnicodeEncodeError:
            if isinstance(s, Exception):
                # An Exception subclass containing non-ASCII data that doesn't
                # know how to print itself properly. We shouldn't raise a
                # further exception.
                return ' '.join([
                    smart_str(arg, encoding, strings_only, errors) for arg in s
                ])
            return unicode(s).encode(encoding, errors)
    elif isinstance(s, unicode):
        return s.encode(encoding, errors)
    elif s and encoding != 'utf-8':
        return s.decode('utf-8', errors).encode(encoding, errors)
    else:
        return s


def b64u_encode(s):
    return urlsafe_b64encode(s).replace('=', '')


def get_grants_config():
    return _get_grants_config()


def invalidate_billing_cache(uid):
    return _invalidate_billing_cache(get_http_pool_manager(), uid)


def _load_broker_retpath_grammars():
    return [UrlGrammar(g) for g in social_config.broker_retpath_grammars]


def _get_broker_retpath_grammars():
    return LazyLoader.get_instance('broker_retpath_grammars')


LazyLoader.register('broker_retpath_grammars', _load_broker_retpath_grammars)

SSO_PASSPORT_PUSH_GRAMMAR = UrlGrammar(
    """
    domain = 'sso.passport.yandex.' yandex_tld
    path = '/push' '/'?
    """,
)
