# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from contextlib import contextmanager
import logging
import Queue
from socket import (
    error as SocketError,
    timeout as SocketTimeout,
)
import sys

import gevent.queue
from passport.backend.social.common.misc import in_gevent
import urllib3
import urllib3.connection
import urllib3.connectionpool
import urllib3.exceptions
import urllib3.poolmanager


logger = logging.getLogger(__name__)


if in_gevent():
    # Эта очередь работает гораздо быстрее, чем пропатченная Queue.LifoQueue
    _LifoQueue = gevent.queue.LifoQueue
else:
    _LifoQueue = Queue.LifoQueue


def _build_pool_key(request_context):
    pool_key = dict(request_context)
    pool_key['scheme'] = pool_key['scheme'].lower()
    pool_key['host'] = pool_key['host'].lower()
    for key, value in pool_key.items():
        if isinstance(value, dict):
            value = frozenset(value.items())
        elif isinstance(value, list):
            value = tuple(value)
        elif isinstance(value, set):
            value = frozenset(value)
        pool_key[key] = value
    return PoolKey(**pool_key)


key_fn_by_scheme = {
    'http': _build_pool_key,
    'https': _build_pool_key,
}


def _create_socket_connection(http_conn):
    """
    Establish a socket connection and set nodelay settings on it.

    :return: New socket connection.
    """
    logger.debug('Create new connection to ' + _format_http_connection_hostname(http_conn))

    extra_kw = {}
    if http_conn.source_address:
        extra_kw['source_address'] = http_conn.source_address

    if http_conn.socket_options:
        extra_kw['socket_options'] = http_conn.socket_options

    try:
        host = http_conn.host_ip or http_conn.host
        conn = urllib3.connection.connection.create_connection((host, http_conn.port), http_conn.timeout, **extra_kw)

    except SocketTimeout:
        raise urllib3.exceptions.ConnectTimeoutError(http_conn, "Connection to %s timed out. (connect timeout=%s)" % (http_conn.format_hostname(), http_conn.timeout))

    except SocketError as e:
        raise urllib3.exceptions.NewConnectionError(http_conn, "Failed to establish a new connection: %s" % e)

    return conn


def _format_http_connection_hostname(http_conn):
    if http_conn.host_ip is not None:
        return '%s (ip_address=%s, id=%s)' % (http_conn.host, http_conn.host_ip, id(http_conn))
    else:
        return '%s (id=%s)' % (http_conn.host, id(http_conn))


@contextmanager
def _domain_invalidation(http_conn):
    logger.debug('Requesting %s' % http_conn.format_hostname())
    try:
        yield
    except (urllib3.exceptions.HTTPError, SocketError):
        if http_conn.dns is not None:
            http_conn.dns.invalidate(http_conn.host, http_conn.host_ip)
        raise


class _HTTPConnection(urllib3.connection.HTTPConnection):
    def __init__(self, host, dns=None, *args, **kwargs):
        super(_HTTPConnection, self).__init__(host, *args, **kwargs)

        self.dns = dns
        self.host_ip = None

    def _new_conn(self):
        if self.dns is not None:
            self.host_ip = self.dns.resolve(self.host)
        return _create_socket_connection(self)

    def format_hostname(self):
        return _format_http_connection_hostname(self)


class _HTTPSConnection(urllib3.connection.HTTPSConnection):
    def __init__(self, host, dns=None, *args, **kwargs):
        super(_HTTPSConnection, self).__init__(host, *args, **kwargs)

        self.dns = dns
        self.host_ip = None

    def _new_conn(self):
        if self.dns is not None:
            self.host_ip = self.dns.resolve(self.host)
        return _create_socket_connection(self)

    def format_hostname(self):
        return _format_http_connection_hostname(self)


class _LoggingConnectionPoolMixin(object):
    def _get_conn(self, *args, **kwargs):
        logger.debug('Get connection from pool: %s' % self.host)
        conn = super(_LoggingConnectionPoolMixin, self)._get_conn(*args, **kwargs)
        fmt_conn = _format_http_connection_hostname(conn)
        logger.debug('Finished to get connection from pool: %s' % fmt_conn)
        return conn

    def _put_conn(self, conn, *args, **kwargs):
        fmt_conn = None
        if conn:
            fmt_conn = _format_http_connection_hostname(conn)
            logger.debug('Put connection to pool (%s)' % fmt_conn)

        retval = super(_LoggingConnectionPoolMixin, self)._put_conn(conn, *args, **kwargs)

        if fmt_conn:
            logger.debug('Finished to put connection to pool (%s)' % fmt_conn)
        return retval


class _FixIssue1758ConnectionPoolMixin(object):
    # Копипаста upstream'а с наложенным фиксом

    def _get_conn(self, timeout=None):
        """
        Get a connection. Will return a pooled connection if one is available.

        If no connections are available and :prop:`.block` is ``False``, then a
        fresh connection is returned.

        :param timeout:
            Seconds to wait before giving up and raising
            :class:`urllib3.exceptions.EmptyPoolError` if the pool is empty and
            :prop:`.block` is ``True``.
        """
        conn = None
        try:
            conn = self.pool.get(block=self.block, timeout=timeout)

        except AttributeError:  # self.pool is None
            raise urllib3.connectionpool.ClosedPoolError(self, "Pool is closed.")

        except urllib3.connectionpool.queue.Empty:
            if self.block:
                raise urllib3.connectionpool.EmptyPoolError(
                    self,
                    "Pool reached maximum size and no more connections are allowed.",
                )
            pass  # Oh well, we'll create a new connection then

        try:
            # If this is a persistent connection, check if it got disconnected
            if conn and urllib3.connectionpool.is_connection_dropped(conn):
                urllib3.connectionpool.log.debug("Resetting dropped connection: %s", self.host)
                conn.close()
                if getattr(conn, 'auto_open', 1) == 0:
                    # This is a proxied connection that has been mutated by
                    # httplib._tunnel() and cannot be reused (since it would
                    # attempt to bypass the proxy)
                    conn = None

            return conn or self._new_conn()
        except:
            self._put_conn(conn)
            raise

    def urlopen(self, method, url, body=None, headers=None, retries=None,
                redirect=True, assert_same_host=True, timeout=urllib3.connectionpool._Default,
                pool_timeout=None, release_conn=None, chunked=False,
                body_pos=None, **response_kw):
        """
        Get a connection from the pool and perform an HTTP request. This is the
        lowest level call for making a request, so you'll need to specify all
        the raw details.

        .. note::

           More commonly, it's appropriate to use a convenience method provided
           by :class:`.RequestMethods`, such as :meth:`request`.

        .. note::

           `release_conn` will only behave as expected if
           `preload_content=False` because we want to make
           `preload_content=False` the default behaviour someday soon without
           breaking backwards compatibility.

        :param method:
            HTTP request method (such as GET, POST, PUT, etc.)

        :param body:
            Data to send in the request body (useful for creating
            POST requests, see HTTPConnectionPool.post_url for
            more convenience).

        :param headers:
            Dictionary of custom headers to send, such as User-Agent,
            If-None-Match, etc. If None, pool headers are used. If provided,
            these headers completely replace any pool-specific headers.

        :param retries:
            Configure the number of retries to allow before raising a
            :class:`~urllib3.exceptions.MaxRetryError` exception.

            Pass ``None`` to retry until you receive a response. Pass a
            :class:`~urllib3.util.retry.Retry` object for fine-grained control
            over different types of retries.
            Pass an integer number to retry connection errors that many times,
            but no other types of errors. Pass zero to never retry.

            If ``False``, then retries are disabled and any exception is raised
            immediately. Also, instead of raising a MaxRetryError on redirects,
            the redirect response will be returned.

        :type retries: :class:`~urllib3.util.retry.Retry`, False, or an int.

        :param redirect:
            If True, automatically handle redirects (status codes 301, 302,
            303, 307, 308). Each redirect counts as a retry. Disabling retries
            will disable redirect, too.

        :param assert_same_host:
            If ``True``, will make sure that the host of the pool requests is
            consistent else will raise HostChangedError. When False, you can
            use the pool on an HTTP proxy and request foreign hosts.

        :param timeout:
            If specified, overrides the default timeout for this one
            request. It may be a float (in seconds) or an instance of
            :class:`urllib3.util.Timeout`.

        :param pool_timeout:
            If set and the pool is set to block=True, then this method will
            block for ``pool_timeout`` seconds and raise EmptyPoolError if no
            connection is available within the time period.

        :param release_conn:
            If False, then the urlopen call will not release the connection
            back into the pool once a response is received (but will release if
            you read the entire contents of the response such as when
            `preload_content=True`). This is useful if you're not preloading
            the response's content immediately. You will need to call
            ``r.release_conn()`` on the response ``r`` to return the connection
            back into the pool. If None, it takes the value of
            ``response_kw.get('preload_content', True)``.

        :param chunked:
            If True, urllib3 will send the body using chunked transfer
            encoding. Otherwise, urllib3 will send the body using the standard
            content-length form. Defaults to False.

        :param int body_pos:
            Position to seek to in file-like body in the event of a retry or
            redirect. Typically this won't need to be set because urllib3 will
            auto-populate the value when needed.

        :param \\**response_kw:
            Additional parameters are passed to
            :meth:`urllib3.response.HTTPResponse.from_httplib`
        """
        if headers is None:
            headers = self.headers

        if not isinstance(retries, urllib3.connectionpool.Retry):
            retries = urllib3.connectionpool.Retry.from_int(retries, redirect=redirect, default=self.retries)

        if release_conn is None:
            release_conn = response_kw.get('preload_content', True)

        # Check host
        if assert_same_host and not self.is_same_host(url):
            raise urllib3.connectionpool.HostChangedError(self, url, retries)

        conn = None
        is_conn_from_pool = False

        # Track whether `conn` needs to be released before
        # returning/raising/recursing. Update this variable if necessary, and
        # leave `release_conn` constant throughout the function. That way, if
        # the function recurses, the original value of `release_conn` will be
        # passed down into the recursive call, and its value will be respected.
        #
        # See issue #651 [1] for details.
        #
        # [1] <https://github.com/shazow/urllib3/issues/651>
        release_this_conn = release_conn

        # Merge the proxy headers. Only do this in HTTP. We have to copy the
        # headers dict so we can safely change it without those changes being
        # reflected in anyone else's copy.
        if self.scheme == 'http':
            headers = headers.copy()
            headers.update(self.proxy_headers)

        # Must keep the exception bound to a separate variable or else Python 3
        # complains about UnboundLocalError.
        err = None

        # Keep track of whether we cleanly exited the except block. This
        # ensures we do proper cleanup in finally.
        clean_exit = False

        # Rewind body position, if needed. Record current position
        # for future rewinds in the event of a redirect/retry.
        body_pos = urllib3.connectionpool.set_file_position(body, body_pos)

        try:
            # Request a connection from the queue.
            timeout_obj = self._get_timeout(timeout)
            conn = self._get_conn(timeout=pool_timeout)
            is_conn_from_pool = True

            conn.timeout = timeout_obj.connect_timeout

            is_new_proxy_conn = self.proxy is not None and not getattr(conn, 'sock', None)
            if is_new_proxy_conn:
                self._prepare_proxy(conn)

            # Make the request on the httplib connection object.
            httplib_response = self._make_request(conn, method, url,
                                                  timeout=timeout_obj,
                                                  body=body, headers=headers,
                                                  chunked=chunked)

            # If we're going to release the connection in ``finally:``, then
            # the response doesn't need to know about the connection. Otherwise
            # it will also try to release it and we'll have a double-release
            # mess.
            response_conn = conn if not release_conn else None

            # Pass method to Response for length checking
            response_kw['request_method'] = method

            # Import httplib's response into our own wrapper object
            response = self.ResponseCls.from_httplib(httplib_response,
                                                     pool=self,
                                                     connection=response_conn,
                                                     retries=retries,
                                                     **response_kw)

            # Everything went great!
            clean_exit = True

        except urllib3.connectionpool.queue.Empty:
            # Timed out by queue.
            raise urllib3.connectionpool.EmptyPoolError(self, "No pool connections are available.")

        except (urllib3.connectionpool.TimeoutError, urllib3.connectionpool.HTTPException, urllib3.connectionpool.SocketError, urllib3.connectionpool.ProtocolError,
                urllib3.connectionpool.BaseSSLError, urllib3.connectionpool.SSLError, urllib3.connectionpool.CertificateError) as e:
            # Discard the connection for these exceptions. It will be
            # replaced during the next _get_conn() call.
            clean_exit = False
            if isinstance(e, (urllib3.connectionpool.BaseSSLError, urllib3.connectionpool.CertificateError)):
                e = urllib3.connectionpool.SSLError(e)
            elif isinstance(e, (urllib3.connectionpool.SocketError, urllib3.connectionpool.NewConnectionError)) and self.proxy:
                e = urllib3.connectionpool.ProxyError('Cannot connect to proxy.', e)
            elif isinstance(e, (urllib3.connectionpool.SocketError, urllib3.connectionpool.HTTPException)):
                e = urllib3.connectionpool.ProtocolError('Connection aborted.', e)

            retries = retries.increment(method, url, error=e, _pool=self,
                                        _stacktrace=sys.exc_info()[2])
            retries.sleep()

            # Keep track of the error for the retry warning.
            err = e

        finally:
            if not clean_exit:
                # We hit some kind of exception, handled or otherwise. We need
                # to throw the connection away unless explicitly told not to.
                # Close the connection, set the variable to None, and make sure
                # we put the None back in the pool to avoid leaking it.
                conn = conn and conn.close()
                release_this_conn = is_conn_from_pool

            if release_this_conn:
                # Put the connection back to be reused. If the connection is
                # expired then it will be None, which will get replaced with a
                # fresh connection during _get_conn.
                self._put_conn(conn)

        if not conn:
            # Try again
            urllib3.connectionpool.log.warning("Retrying (%r) after connection broken by '%r': %s", retries, err, url)
            return self.urlopen(method, url, body, headers, retries,
                                redirect, assert_same_host,
                                timeout=timeout, pool_timeout=pool_timeout,
                                release_conn=release_conn, body_pos=body_pos,
                                **response_kw)

        def drain_and_release_conn(response):
            try:
                # discard any remaining response body, the connection will be
                # released back to the pool once the entire response is read
                response.read()
            except (urllib3.connectionpool.TimeoutError, urllib3.connectionpool.HTTPException, urllib3.connectionpool.SocketError, urllib3.connectionpool.ProtocolError,
                    urllib3.connectionpool.BaseSSLError, urllib3.connectionpool.SSLError):
                pass

        # Handle redirect?
        redirect_location = redirect and response.get_redirect_location()
        if redirect_location:
            if response.status == 303:
                method = 'GET'

            try:
                retries = retries.increment(method, url, response=response, _pool=self)
            except urllib3.connectionpool.MaxRetryError:
                if retries.raise_on_redirect:
                    # Drain and release the connection for this response, since
                    # we're not returning it to be released manually.
                    drain_and_release_conn(response)
                    raise
                return response

            # drain and return the connection to the pool before recursing
            drain_and_release_conn(response)

            retries.sleep_for_retry(response)
            urllib3.connectionpool.log.debug("Redirecting %s -> %s", url, redirect_location)
            return self.urlopen(
                method, redirect_location, body, headers,
                retries=retries, redirect=redirect,
                assert_same_host=assert_same_host,
                timeout=timeout, pool_timeout=pool_timeout,
                release_conn=release_conn, body_pos=body_pos,
                **response_kw)

        # Check if we should retry the HTTP response.
        has_retry_after = bool(response.getheader('Retry-After'))
        if retries.is_retry(method, response.status, has_retry_after):
            try:
                retries = retries.increment(method, url, response=response, _pool=self)
            except urllib3.connectionpool.MaxRetryError:
                if retries.raise_on_status:
                    # Drain and release the connection for this response, since
                    # we're not returning it to be released manually.
                    drain_and_release_conn(response)
                    raise
                return response

            # drain and return the connection to the pool before recursing
            drain_and_release_conn(response)

            retries.sleep(response)
            urllib3.connectionpool.log.debug("Retry: %s", url)
            return self.urlopen(
                method, url, body, headers,
                retries=retries, redirect=redirect,
                assert_same_host=assert_same_host,
                timeout=timeout, pool_timeout=pool_timeout,
                release_conn=release_conn,
                body_pos=body_pos, **response_kw)

        return response


class _HTTPConnectionPool(
    _LoggingConnectionPoolMixin,
    _FixIssue1758ConnectionPoolMixin,
    urllib3.HTTPConnectionPool,
):
    ConnectionCls = _HTTPConnection
    QueueCls = _LifoQueue

    def __init__(self, *args, **kwargs):
        kwargs.pop('forbidden_ssl_versions', None)
        super(_HTTPConnectionPool, self).__init__(*args, **kwargs)

    def _make_request(self, conn, method, url, **kwargs):
        with _domain_invalidation(conn):
            return super(_HTTPConnectionPool, self)._make_request(conn, method, url, **kwargs)


class _HTTPSConnectionPool(
    _LoggingConnectionPoolMixin,
    _FixIssue1758ConnectionPoolMixin,
    urllib3.HTTPSConnectionPool,
):
    ConnectionCls = _HTTPSConnection
    QueueCls = _LifoQueue

    def __init__(self, *args, **kwargs):
        self._forbidden_ssl_versions = kwargs.pop('forbidden_ssl_versions', None)
        super(_HTTPSConnectionPool, self).__init__(*args, **kwargs)

    def _make_request(self, conn, method, url, **kwargs):
        with _domain_invalidation(conn):
            return super(_HTTPSConnectionPool, self)._make_request(conn, method, url, **kwargs)

    def _validate_conn(self, conn):
        super(_HTTPSConnectionPool, self)._validate_conn(conn)

        proto = conn.sock.version()
        logger.debug('SSL protocol version: %s' % proto)
        if self._forbidden_ssl_versions and proto in self._forbidden_ssl_versions:
            raise urllib3.exceptions.SSLError('Forbidden protocol version: %s' % proto)


class PoolManager(urllib3.PoolManager):
    def __init__(self, *args, **kwargs):
        super(PoolManager, self).__init__(*args, **kwargs)
        self.pool_classes_by_scheme = {
            'http': _HTTPConnectionPool,
            'https': _HTTPSConnectionPool,
        }
        self.key_fn_by_scheme = key_fn_by_scheme.copy()


class ProxyManager(urllib3.ProxyManager):
    def __init__(self, *args, **kwargs):
        super(ProxyManager, self).__init__(*args, **kwargs)
        self.pool_classes_by_scheme = {
            'http': _HTTPConnectionPool,
            'https': _HTTPSConnectionPool,
        }
        self.key_fn_by_scheme = key_fn_by_scheme.copy()


class PoolKey(tuple):
    def __new__(cls, **kwargs):
        items = list()
        key_to_idx = dict()
        for key in sorted(kwargs):
            items.append((key, kwargs[key]))
            key_to_idx[key] = len(items) - 1
        self = super(PoolKey, cls).__new__(cls, items)
        self.__key_to_idx = key_to_idx
        return self

    def __getattr__(self, name):
        try:
            return super(PoolKey, self).__getattr__(name)
        except AttributeError:
            if name not in self.__key_to_idx:
                raise
            return self[self.__key_to_idx[name]][1]
