# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from uuid import uuid4

from passport.backend.core import Undefined
from passport.backend.core.models.base.fields import Field
from passport.backend.core.models.base.parse import (
    parse_boolean_field,
    parse_integer_field,
)
from passport.backend.social.common.db.execute import execute
from passport.backend.social.common.db.schemas import (
    application_attribute_table,
    application_index_attribute_table,
    application_table,
)
from passport.backend.social.common.eav import (
    EavConfiguration,
    EavDatabaseReader,
    EavDatabaseSerializer,
    EavModelDatabaseSerializer,
    EavSimpleModelConverter,
)
from passport.backend.social.common.models.base import Model
from passport.backend.social.common.models.parse import parse_list_field
from passport.backend.social.common.serialize import (
    get_boolean_db_value,
    get_integer_db_value,
    get_not_empty_string_db_value,
    ValueSerializationError,
)
from sqlalchemy import and_ as sql_and


logger = logging.getLogger(__name__)


def build_application_name():
    return _build_application_name()


def _build_application_name():
    return uuid4().hex


def parse_provider_id_field(field_name):
    def _parse(data, *args):
        from passport.backend.social.common.provider_settings import providers

        processed, value = False, Undefined
        provider_id = data.get(field_name)
        if provider_id:
            provider_info = providers.get_provider_info_by_id(int(provider_id))
            if provider_info:
                processed, value = True, provider_info
        return processed, value
    return _parse


def parse_user_param_field(field_name):
    def _parse(data, *args):
        from passport.backend.social.common.misc import UserParamDescriptor
        try:
            user_param = UserParamDescriptor.from_str(data.get(field_name, ''))
            return True, user_param
        except ValueError:
            return False, Undefined
    return _parse


def get_tld_db_value(value):
    if value is Undefined or value is None:
        return
    if not isinstance(value, (list, tuple, set)):
        raise ValueSerializationError()
    if not value:
        return
    for tld in value:
        if not isinstance(tld, basestring):
            raise ValueSerializationError()
        if ';' in tld:
            # TODO Здесь нужно реализовать экранирование `;`
            raise NotImplementedError()
    if isinstance(value, set):
        ordered_tlds = sorted(value)
    else:
        ordered_tlds = value
    return ';'.join(ordered_tlds)


def get_provider_db_value(value):
    if value is Undefined or value is None:
        return
    try:
        provider_id = value['id']
    except (TypeError, KeyError):
        raise ValueSerializationError()
    try:
        provider_id = int(provider_id)
    except (TypeError, ValueError):
        raise ValueSerializationError()
    return provider_id


def get_user_param_db_value(value):
    if value is Undefined or value is None:
        return
    from passport.backend.social.common.misc import UserParamDescriptor
    if not isinstance(value, UserParamDescriptor):
        return
    return str(value)


class Application(Model):
    # Internal application id
    identifier = Field(parse_integer_field('application_id'))

    # External application id
    name = Field('application_name')

    provider = Field(parse_provider_id_field('provider_id'))
    # Provider client id
    id = Field('provider_client_id')
    # Когда несколько приложений используют одинаковый provider_client_id, то
    # удобно хранить в этом поле настоящий provider_client_id, а в поле id
    # уникальную строку, чтобы не нарушать ограничений БД
    custom_provider_client_id = Field('custom_provider_client_id')

    # Provider client secret
    secret = Field('secret')

    # Авторизация на default scope будет запрошена даже, если потребитель не
    # запросил этого явно
    default_scope = Field('default_scope')
    # Если потребитель запросит авторизацию на скоуп, которого нет в allowed
    # scope, то такой скоуп будет исключён из запроса
    allowed_scope = Field('allowed_scope')

    authorization_url = Field('authorization_url')
    token_url = Field('token_url')
    refresh_token_url = Field('refresh_token_url')
    app_server_key = Field('app_server_key')
    key = Field('key')
    default = Field(parse_boolean_field('default'))
    domain = Field('domain')
    tld = Field(parse_list_field('tld'))
    engine_id = Field('engine_id')
    group_id = Field('group_id')
    is_third_party = Field(parse_boolean_field('is_third_party'))
    related_yandex_client_id = Field('related_yandex_client_id')
    related_yandex_client_secret = Field('related_yandex_client_secret')
    request_from_intranet_allowed = Field(parse_boolean_field('request_from_intranet_allowed'))

    display_name = Field('display_name')

    authorize_user_param = Field(parse_user_param_field('authorize_user_param'))
    token_user_param = Field(parse_user_param_field('token_user_param'))

    apple_team_id = Field('apple_team_id')
    apple_jwt_certificate_id = Field('apple_jwt_certificate_id')

    def get_display_name_or_default(self):
        return self.display_name or self.name


application_eav_configuration = EavConfiguration(
    eav_attributes=dict(
        secret=3,
        default_scope=4,
        authorization_url=5,
        token_url=6,
        refresh_token_url=7,
        app_server_key=8,
        key=9,
        default=10,
        domain=11,
        engine_id=12,
        is_third_party=14,
        related_yandex_client_id=16,
        related_yandex_client_secret=17,
        tld=18,
        request_from_intranet_allowed=19,
        display_name=20,
        allowed_scope=22,
        authorize_user_param=23,
        token_user_param=24,
        apple_team_id=25,
        apple_jwt_certificate_id=26,
    ),
    eav_table=application_attribute_table,

    eav_index_attributes=dict(
        group_id=13,
        custom_provider_client_id=21,
    ),
    eav_index_table=application_index_attribute_table,

    index_attributes=[
        'application_id',
        'provider_id',
        'provider_client_id',
        'application_name',
    ],
    index_table=application_table,
)


class ApplicationDatabaseReader(EavDatabaseReader):
    eav_configuration = application_eav_configuration

    def load_by_group_ids(self, values):
        table = self.eav_configuration.eav_index_table
        attr_type = self.eav_configuration.get_type_from_name('group_id')
        query = (
            table.select()
            .where(
                sql_and(
                    table.c.type == attr_type,
                    table.c.value.in_(values),
                ),
            )
        )
        eav_result = execute(self.database, query).fetchall()
        ids = [er['application_id'] for er in eav_result]
        return self.load_by_application_ids(ids)


class ApplicationDatabaseSerializer(EavDatabaseSerializer):
    eav_configuration = application_eav_configuration


class ApplicationModelDatabaseSerializer(EavModelDatabaseSerializer):
    model = Application
    database_serializer_class = ApplicationDatabaseSerializer

    eav_model_converters = [
        EavSimpleModelConverter('identifier', get_integer_db_value, 'application_id'),
        EavSimpleModelConverter('name', get_not_empty_string_db_value, 'application_name'),
        EavSimpleModelConverter('provider', get_provider_db_value, 'provider_id'),
        EavSimpleModelConverter('id', get_not_empty_string_db_value, 'provider_client_id'),
        EavSimpleModelConverter('custom_provider_client_id', get_not_empty_string_db_value),
        EavSimpleModelConverter('secret', get_not_empty_string_db_value),
        EavSimpleModelConverter('default_scope', get_not_empty_string_db_value),
        EavSimpleModelConverter('allowed_scope', get_not_empty_string_db_value),
        EavSimpleModelConverter('authorization_url', get_not_empty_string_db_value),
        EavSimpleModelConverter('token_url', get_not_empty_string_db_value),
        EavSimpleModelConverter('refresh_token_url', get_not_empty_string_db_value),
        EavSimpleModelConverter('app_server_key', get_not_empty_string_db_value),
        EavSimpleModelConverter('key', get_not_empty_string_db_value),
        EavSimpleModelConverter('default', get_boolean_db_value),
        EavSimpleModelConverter('domain', get_not_empty_string_db_value),
        EavSimpleModelConverter('tld', get_tld_db_value),
        EavSimpleModelConverter('engine_id', get_not_empty_string_db_value),
        EavSimpleModelConverter('group_id', get_not_empty_string_db_value),
        EavSimpleModelConverter('is_third_party', get_boolean_db_value),
        EavSimpleModelConverter('related_yandex_client_id', get_not_empty_string_db_value),
        EavSimpleModelConverter('related_yandex_client_secret', get_not_empty_string_db_value),
        EavSimpleModelConverter('request_from_intranet_allowed', get_boolean_db_value),
        EavSimpleModelConverter('display_name', get_not_empty_string_db_value),
        EavSimpleModelConverter('authorize_user_param', get_user_param_db_value),
        EavSimpleModelConverter('token_user_param', get_user_param_db_value),
        EavSimpleModelConverter('apple_team_id', get_not_empty_string_db_value),
        EavSimpleModelConverter('apple_jwt_certificate_id', get_not_empty_string_db_value),
    ]


class ApplicationGroupId(object):
    passport = 'passport'
    station = 'bass'
