# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.social.common.exception import SocialException
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import UserAgent


logger = logging.getLogger(__name__)


def invalidate_billing_cache(http_pool_manager, uid, fail_safe=True):
    if not social_config.invalidate_billing_binding_cache:
        return
    billing_api = BillingApi(http_pool_manager)
    try:
        billing_api.invalidate_account_bindings(uid)
    except TemporaryBillingApiError:
        if not fail_safe:
            raise
        # Билдер уже записал в лог об отказе из-за временной ошибки, больше
        # ничего не делаем.
    except BillingApiError as e:
        if not fail_safe:
            raise
        logger.warning('Failed to invalidate account bindings for uid %s in Billing (%s)' % (uid, str(e)))


class BillingApiError(SocialException):
    code = 'generic-billing-error'


class TemporaryBillingApiError(BillingApiError):
    code = 'temporary-billing-error'


class ParsingBillingApiError(BillingApiError):
    code = 'parsing-billing-error'


class BillingApi(JsonParserMixin, BaseBuilder):
    base_error_class = BillingApiError
    temporary_error_class = TemporaryBillingApiError
    parser_error_class = ParsingBillingApiError

    def __init__(self, http_pool_manager):
        retries = social_config.billing_http_api_retries
        timeout = social_config.billing_http_api_timeout

        useragent = UserAgent(
            # BaseBuilder переопределяет допустимое время на каждый вызов
            timeout=timeout,
            # BaseBuilder делает повторные попытки сам
            retries=1,
            pool_manager=http_pool_manager,
            in_passport_builder=True,
        )

        graphite_logger = GraphiteLogger(
            logging.getLogger('graphite.useragent'),
            service='billing_xmlrpc',
        )

        super(BillingApi, self).__init__(
            url=social_config.billing_http_api_url,
            useragent=useragent,
            timeout=timeout,
            retries=retries,
            logger=logger,
            graphite_logger=graphite_logger,
        )

        self._socialism_token = social_config.billing_http_api_service_token

    def invalidate_account_bindings(self, uid):
        def detect_error(response, raw_response):
            status = response.get('status')
            if status != 'success':
                if status == 'error' and response.get('status_code') == 'unknown_error':
                    raise TemporaryBillingApiError()
                raise BillingApiError('Unexpected response: %s' % raw_response.content)

        path = '/trust-payments/v2/passport/%s/invalidate' % uid

        self._request_with_retries_simple(
            error_detector=detect_error,
            parser=self.parse_json,
            method='POST',
            url_suffix=path,
            headers={'X-Service-Token': self._socialism_token},
        )
