# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import contextlib
import logging

from passport.backend.core.builders.blackbox import (
    Blackbox as _Blackbox,
    exceptions as passport_exceptions,
)
from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import SocialException
from passport.backend.social.common.misc import X_TOKEN_SCOPE
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import UserAgent


BLACKBOX_PHONE_ATTRIBUTES = (
    'phones.default',
    'phones.secure',
)

BLACKBOX_PHONE_EXTENDED_ATTRIBUTES = (
    'number',
    'created',
    'bound',
    'confirmed',
    'admitted',
    'secured',
)


def add_phone_arguments(**blackbox_kwargs):
    attributes = blackbox_kwargs.get('attributes')
    if attributes is None:
        attributes = tuple()
    blackbox_kwargs.update({
        'phones': 'all',
        'need_aliases': True,
        'attributes': tuple(attributes) + BLACKBOX_PHONE_ATTRIBUTES,
        'need_current_phone_bindings': True,
        'need_unbound_phone_bindings': True,
        'phone_attributes': BLACKBOX_PHONE_EXTENDED_ATTRIBUTES,
    })
    return blackbox_kwargs


def check_session_cookie(sessionid_response):
    if sessionid_response.get('error', 'OK') != 'OK':
        raise BlackboxInvalidSessionidError(sessionid_response['error'])
    if sessionid_response['status'] not in {'VALID', 'NEED_RESET'}:
        raise BlackboxInvalidSessionidError()


def check_oauth_response(oauth_response):
    if oauth_response.get('error', 'OK') != 'OK':
        raise BlackboxOauthTokenInvalidError(oauth_response['error'])
    if oauth_response['status'] not in {'VALID'}:
        raise BlackboxOauthTokenInvalidError()


def check_token_suitable_for_binding(token, real_client_id):
    _check_oauth_token_suitable_for_binding(real_client_id, token.scopes)


def check_oauth_response_suitable_for_binding(oauth_response):
    client_id = (
        oauth_response
        .get('oauth', dict())
        .get('client_id')
    )
    token_scopes = (
        oauth_response
        .get('oauth', dict())
        .get('scope', list())
    )
    _check_oauth_token_suitable_for_binding(client_id, token_scopes)


def authenticated():
    return request_ctx.authenticated


@contextlib.contextmanager
def consider_authenticated():
    request_ctx.authenticated, old = True, request_ctx.authenticated
    try:
        yield
    finally:
        request_ctx.authenticated = old


def _check_oauth_token_suitable_for_binding(client_id, scopes):
    if _are_scopes_suitable_for_binding(scopes):
        return
    if providers.get_application_by_provider_app_id(Yandex.id, client_id):
        return
    raise BlackboxOauthTokenInvalidError('Not enough scope: %s' % scopes)


def _are_scopes_suitable_for_binding(token_scopes):
    for s in social_config.oauth_scopes_required:
        if s in token_scopes:
            return True
    return X_TOKEN_SCOPE in token_scopes


class Blackbox(_Blackbox):
    def __init__(self, http_pool_manager=None):
        url = social_config.blackbox_url
        if social_config.broker_stress_mode:
            url = social_config.broker_stress_emulator_blackbox_url  # pragma: no cover

        retries = social_config.blackbox_retries
        timeout = social_config.blackbox_timeout

        useragent = UserAgent(
            # BaseBuilder переопределяет допустимое время на каждый вызов
            timeout=timeout,
            # BaseBuilder делает повторные попытки сам
            retries=1,
            pool_manager=http_pool_manager,
            in_passport_builder=True,
        )

        graphite_logger = GraphiteLogger(
            logging.getLogger('graphite.useragent'),
            service='blackbox',
        )

        super(Blackbox, self).__init__(
            blackbox=url,
            useragent=useragent,
            timeout=timeout,
            retries=retries,
            graphite_logger=graphite_logger,
            use_tvm=True,

        )

    def build_login_request(self, **kwargs):
        self._update_userinfo_kwargs(kwargs)
        return super(Blackbox, self).build_login_request(**kwargs)

    def build_oauth_request(self, **kwargs):
        self._update_userinfo_kwargs(kwargs)
        return super(Blackbox, self).build_oauth_request(**kwargs)

    def build_sessionid_request(self, **kwargs):
        self._update_userinfo_kwargs(kwargs)
        return super(Blackbox, self).build_sessionid_request(**kwargs)

    def build_userinfo_request(self, **kwargs):
        self._update_userinfo_kwargs(kwargs)
        return super(Blackbox, self).build_userinfo_request(**kwargs)

    def _update_userinfo_kwargs(self, kwargs):
        for arg_name, default_value in dict(
            get_hidden_aliases=False,
            need_public_name=False,
        ).items():
            kwargs.setdefault(arg_name, default_value)


class BlackboxOauthTokenInvalidError(SocialException):
    pass


class BlackboxInvalidSessionidError(SocialException):
    pass


BlackboxTemporaryError = passport_exceptions.BlackboxTemporaryError
BlackboxInvalidResponseError = passport_exceptions.BlackboxInvalidResponseError
BlackboxUnknownError = passport_exceptions.BlackboxUnknownError
