# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import logging

from passport.backend.core.builders.passport import (
    exceptions as passport_exceptions,
    passport,
)
from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.core.tvm.tvm_credentials_manager import get_tvm_credentials_manager
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import UserAgent


class Passport(passport.Passport):
    def __init__(self, http_pool_manager=None):
        useragent = UserAgent(
            in_passport_builder=True,
            pool_manager=http_pool_manager,
            # BaseBuilder делает повторные попытки сам
            retries=1,
            # BaseBuilder переопределяет допустимое время на каждый вызов
            timeout=social_config.passport_api_timeout,
        )

        graphite_logger = GraphiteLogger(
            logging.getLogger('graphite.useragent'),
            service='passport',
        )

        super(Passport, self).__init__(
            consumer=social_config.passport_api_consumer,
            graphite_logger=graphite_logger,
            retries=social_config.passport_api_retries,
            timeout=social_config.passport_api_timeout,
            tvm_credentials_manager=get_tvm_credentials_manager(),
            url=social_config.passport_api_url,
            use_tvm=True,
            useragent=useragent,
        )

    def drop_phone(self, uid, phone_id):
        data = {
            'phoneid': str(phone_id),
            'sender': self.consumer,
            'uid': str(uid),
        }
        return self._request_with_retries_simple(
            data=data,
            error_detector=self.detect_drop_phone_error,
            http_error_handler=passport.error_handler,
            method='POST',
            parser=self.parse_json,
            url_suffix='/yasms/api/dropphone',
        )

    def detect_drop_phone_error(self, response, raw_response):
        error_code = response.get('error')
        if error_code == 'INTERROR':
            raise PassportTemporaryError()
        elif error_code:
            raise PassportPermanentError()

    def account_options(self, uid, global_logout=None):
        params = {}

        if global_logout is not None:
            params['global_logout'] = passport.to_bool(global_logout)

        return self._base_request(
            url_suffix='2/account/%s/options/' % uid,
            http_method='POST',
            error_detector=passport.bundle_api_error_detector,
            post_args=params,
        )


BasePassportError = passport_exceptions.BasePassportError
PassportPhoneOperationExistsError = passport_exceptions.PassportPhoneOperationExistsError
PassportAccountNotFoundError = passport_exceptions.PassportAccountNotFoundError
PassportAccountDisabledError = passport_exceptions.PassportAccountDisabledError
PassportTemporaryError = passport_exceptions.PassportTemporaryError
PassportTrackNotFoundError = passport_exceptions.PassportTrackNotFoundError
PassportTrackInvalidStateError = passport_exceptions.PassportTrackInvalidStateError
PassportPermanentError = passport_exceptions.PassportPermanentError
PassportUserNotVerifiedError = passport_exceptions.PassportUserNotVerifiedError
