# -*- coding: utf-8 -*-

import gevent
import gevent.monkey


threading_current_thread = gevent.monkey.get_original('threading', 'current_thread')


class _ContextAttribute(object):
    def __init__(self, attr_name):
        self._attr_name = attr_name

    def __get__(self, context, context_cls=None):
        return context._get(self._attr_name)

    def __set__(self, context, value):
        context._put(self._attr_name, value)


class _Context(object):
    application = _ContextAttribute('application')
    authenticated = _ContextAttribute('authenticated')
    handler_id = _ContextAttribute('handler_id')
    provider = _ContextAttribute('provider')
    request_id = _ContextAttribute('request_id')
    task = _ContextAttribute('task')
    task_id = _ContextAttribute('task_id')
    grants_context = _ContextAttribute('grants_context')

    def __init__(self):
        self._dict = dict()

    def _get(self, attr_name, default=None):
        pid = self._build_process_id()
        if pid not in self._dict:
            return default
        return self._dict[pid].get(attr_name, default)

    def _put(self, attr_name, value):
        pid = self._build_process_id()
        if pid not in self._dict:
            self._dict[pid] = dict()
        self._dict[pid][attr_name] = value

    def clear(self):
        pid = self._build_process_id()
        if pid in self._dict:
            del self._dict[pid]

    def save(self):
        pid = self._build_process_id()
        if pid in self._dict:
            return self._dict[pid]
        else:
            return dict()

    def load(self, data):
        pid = self._build_process_id()
        if pid not in self._dict:
            self._dict[pid] = dict()
        self._dict[pid].clear()
        self._dict[pid].update(data)

    def _build_process_id(self):
        return (threading_current_thread(), gevent.getcurrent())


request_ctx = _Context()
