# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from collections import OrderedDict


class DirectedGraph(object):
    def __init__(self):
        self._nodes = set()

    def add_node(self, value):
        node = _GraphNode(value)
        self._nodes.add(node)
        return node

    def has_node(self, node):
        return node in self._nodes

    def add_edge(self, src, dst, value=None):
        assert self.has_node(src)
        assert self.has_node(dst)
        edge = _DirectedEdge(src=src, dst=dst, value=value)
        src.add_edge(edge)
        dst.add_edge(edge)
        return edge

    def remove_node(self, node):
        if not self.has_node(node):
            return
        for edge in node.out_edges:
            edge.dst.remove_in_edge(node)
        for edge in node.in_edges:
            edge.src.remove_out_edge(node)
        self._nodes.remove(node)

    def traverse_bfs(self, start, edge_key=None):
        """
        Входные параметры
          edge_key (опц.)
            Функция, которая ставит в соответствие каждому ребру его
            "не важность". Рёбра исходящие из одной вершины будут обходится в
            порядке возрастания "не важности".
        """
        if edge_key is None:
            edge_key = lambda e: e.value

        queue = [(start, [])]
        used = set([start])
        while queue:
            node, path = queue.pop(0)
            yield node, path
            if self.has_node(node):
                for edge in sorted(node.out_edges, key=edge_key):
                    if edge.dst not in used:
                        used.add(edge.dst)
                        queue.append((edge.dst, path + [edge]))


class _GraphNode(object):
    def __init__(self, value):
        self.value = value
        self._out_edges = OrderedDict()
        self._in_edges = dict()

    @property
    def out_edges(self):
        return self._out_edges.values()

    @property
    def in_edges(self):
        return self._in_edges.values()

    def add_edge(self, edge):
        if edge.src is self:
            self._out_edges[edge.dst] = edge
        elif edge.dst is self:
            self._in_edges[edge.src] = edge
        else:
            assert False  # pragma: no cover

    def remove_in_edge(self, src_node):
        del self._in_edges[src_node]

    def remove_out_edge(self, dst_node):
        del self._out_edges[dst_node]


class _DirectedEdge(object):
    def __init__(self, src, dst, value):
        self.src = src
        self.dst = dst
        self.value = value
