# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.utils import is_read_only_db_query
from passport.backend.social.common.exception import DatabaseError
from sqlalchemy.exc import (
    DBAPIError,
    OperationalError,
    TimeoutError as PoolTimeoutError,
)


logger = logging.getLogger(__name__)
subprocess = logging.getLogger('subprocess.error')
graphite = GraphiteLogger(logging.getLogger('graphite.database'))


FAIL_DB_RESPONSE = 'fail'
TIMEOUT_DB_RESPONSE = 'timeout'
POOL_TIMEOUT_DB_RESPONSE = 'pool_timeout'
SUCCESS_DB_RESPONSE = 'success'
WRITE_QUERY_TYPE = 'write'
READ_QUERY_TYPE = 'read'


def execute(engine, query, retries=None, timeout=None):
    retries = retries if retries is not None else engine.reconnect_retries

    while True:
        start_time = now.f()

        duration_record = _DatabaseRequestDuration()

        try:
            result = _execute(engine, query, duration_record)
        except DBAPIError as e:
            retries -= 1
            logger.warning('DB error during query: %s' % str(e))

            duration_record.total = now.f() - start_time
            _log_error_to_graphite(query, engine, retries, duration_record, e)

            if retries <= 0:
                raise DatabaseError()
        except PoolTimeoutError as e:
            logger.warning('DB connection pool exhausted: %s' % str(e))
            duration_record.total = now.f() - start_time
            _log_error_to_graphite(query, engine, 0, duration_record, e)
            raise DatabaseError()
        except Exception as e:
            duration_record.total = now.f() - start_time
            _log_error_to_graphite(query, engine, 0, duration_record, e)
            raise
        else:
            duration_record.total = now.f() - start_time
            _log_success_to_graphite(query, engine, 0, duration_record)
            return result


def _log_success_to_graphite(query, engine, retries_left, duration_record):
    log_record = _build_common_graphite_record(query, engine, retries_left, duration_record)
    log_record['response'] = SUCCESS_DB_RESPONSE
    graphite.log(**log_record)


def _log_error_to_graphite(query, engine, retries_left, duration_record, exception):
    """
    retries_left — число попыток, которое будет сделано после данной попытки
    """
    log_record = _build_common_graphite_record(query, engine, retries_left, duration_record)

    if _exception_is_database_timeout(exception):
        log_record['response'] = TIMEOUT_DB_RESPONSE
    elif isinstance(exception, PoolTimeoutError):
        log_record['response'] = POOL_TIMEOUT_DB_RESPONSE
    else:
        log_record['response'] = FAIL_DB_RESPONSE

    graphite.log(**log_record)


def _build_common_graphite_record(query, engine, retries_left, duration_record):
    log_record = dict(
        tskv_format='social-database-log',
        srv_hostname=engine.url.host,
        service='database',
        duration=duration_record.total,
        connect_duration=duration_record.connect,
        request_duration=duration_record.request,
        retries_left=retries_left,
        request_id=request_ctx.request_id,
    )
    if is_read_only_db_query(query):
        log_record['query_type'] = READ_QUERY_TYPE
    else:
        log_record['query_type'] = WRITE_QUERY_TYPE
    return log_record


def _exception_is_database_timeout(exception):
    return (
        isinstance(exception, OperationalError) and
        str(exception).startswith('(_mysql_exceptions.OperationalError) (2013,')
    )


def _timed_execute(connection, query, duration):
    started_time = now.f()
    try:
        return connection.execute(query)
    finally:
        duration.request = now.f() - started_time


def _execute(engine, query, duration):
    started_time = now.f()
    try:
        logger.debug('Get database connection from pool: %s' % unicode(engine))
        connection = engine.contextual_connect(close_with_result=True)
        logger.debug('Finished to get database connection from pool: %s (id=%s)' % (unicode(engine), id(connection)))
    finally:
        duration.connect = now.f() - started_time

    return _timed_execute(connection, query, duration)


class _DatabaseRequestDuration(object):
    def __init__(self):
        self.total = None
        self.connect = None
        self.request = None
