# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common.db.types import (
    DATE,
    TIMESTAMP,
)
from sqlalchemy.schema import (
    Column,
    ForeignKey,
    Index,
    MetaData,
    Table,
)
from sqlalchemy.sql.functions import current_timestamp
from sqlalchemy.types import (
    BigInteger,
    Enum,
    Integer,
    SmallInteger,
    String,
    VARCHAR,
)


logger = logging.getLogger('passport.backend.social.common.db')

metadata = MetaData()


profile_table = Table(
    'profile',
    metadata,
    Column('profile_id', Integer, primary_key=True, nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('provider_id', SmallInteger, nullable=False),
    Column('allow_auth', SmallInteger, nullable=False, default=0),

    # Для ФБ здесь может быть как business_uid, так и глобальный uid.
    # Так получилось из-за того, что в момент замены глобальных
    # uid'ов пользователей на business_uid'ы, некоторые пользователи забрали
    # доступ у приложений Яндекса.
    Column('userid', String(255), nullable=False),

    Column('username', String(255), nullable=False, default=''),
    Column('created', TIMESTAMP(), default=current_timestamp(), nullable=False),
    # verified означает что Социализм проверил профиль (когда то профили
    # вливались из других источников и они считались непроверенными).
    Column('verified', TIMESTAMP(), server_default='0000-00-00 00:00:00', nullable=False),
    Column('confirmed', TIMESTAMP(), server_default='0000-00-00 00:00:00', nullable=False),
    Column('referer', Integer, nullable=False, default=0),
    Column('yandexuid', String(40), nullable=False, default=''),
    Index('unique_key', 'provider_id', 'userid', 'uid', unique=True),
    Index('uid', 'uid'),
    sqlite_autoincrement=True,
)

person_table = Table(
    'person',
    metadata,
    Column('profile_id', BigInteger, ForeignKey(profile_table.c.profile_id), primary_key=True, autoincrement=False, nullable=False),
    Column('firstname', String(255), nullable=False, default=''),
    Column('lastname', String(255), nullable=False, default=''),
    Column('nickname', String(255), nullable=False, default=''),
    Column('email', String(255), nullable=False, default=''),
    Column('phone', String(255), nullable=False, default=''),
    Column('country', String(255), nullable=False, default=''),
    Column('city', String(255), nullable=False, default=''),
    Column('birthday', DATE, nullable=False, server_default='0000-00-00'),
    Column('gender', Enum('', 'm', 'f'), nullable=False, default=''),
    sqlite_autoincrement=True,
)

token_table = Table(
    'token',
    metadata,
    Column('token_id', Integer, nullable=False, primary_key=True),
    Column('uid', BigInteger, nullable=False),
    Column('profile_id', BigInteger, ForeignKey(profile_table.c.profile_id), nullable=True),
    Column('application_id', BigInteger, nullable=False),
    Column('value', String(255), nullable=False),
    Column('value_hash', String(255), nullable=False),
    Column('secret', String(255), nullable=False, default=''),
    Column('scope', String(255), nullable=False, default=''),
    Column('created', TIMESTAMP(), nullable=False, default=current_timestamp()),
    Column('verified', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('confirmed', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('expired', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Index('token_uid', 'uid', 'application_id', 'value', unique=True),
    Index('profile_app', 'profile_id', 'application_id', unique=False),
    Index('token_hash_uid', 'uid', 'application_id', 'value_hash', unique=True),
    sqlite_autoincrement=True,
)

sub_table = Table(
    'subscription',
    metadata,
    Column('profile_id', BigInteger, ForeignKey(profile_table.c.profile_id), nullable=False, primary_key=True, autoincrement=False),
    Column('sid', SmallInteger, nullable=False, primary_key=True, autoincrement=False),
    Column('value', SmallInteger, nullable=False),
    sqlite_autoincrement=True,
)

business_application_map_table = Table(
    'business_application_map',
    metadata,
    Column('id', Integer, nullable=False, primary_key=True),
    Column('business_id', SmallInteger, nullable=False, default='0'),

    Column('business_token', VARCHAR(255), nullable=False, default=''),

    Column('application_id', BigInteger, nullable=False, default='0'),

    # Либо идентфикатор пользователя в приложении ФБ, либо идентификатор
    # пользователя, который был у него до появления пространств идентификаторов
    # пользователя в приложениях.
    Column('userid', VARCHAR(255), nullable=False, default=''),

    Column('ts', TIMESTAMP, nullable=False, default=current_timestamp()),
    Index('id', 'id', unique=True),
    Index('business_id', 'business_id', 'business_token', 'application_id', 'userid', unique=True),
    Index('userid', 'userid', 'application_id'),
    sqlite_autoincrement=True,
)

refresh_token_table = Table(
    'refresh_token',
    metadata,
    Column('refresh_token_id', Integer, nullable=False, primary_key=True),
    Column('value', String(255), nullable=False),
    Column('expired', TIMESTAMP(), nullable=False),
    Column('token_id', BigInteger, nullable=False, unique=True),
    sqlite_autoincrement=True,
)

application_attribute_table = Table(
    'application_attribute',
    metadata,
    Column('application_id', BigInteger, nullable=False, primary_key=True, autoincrement=False),
    Column('type', SmallInteger, nullable=False, primary_key=True, autoincrement=False),
    Column('value', String(2048), nullable=False),
)

application_index_attribute_table = Table(
    'application_index_attribute',
    metadata,
    Column('application_id', BigInteger, nullable=False, primary_key=True, autoincrement=False),
    Column('type', SmallInteger, nullable=False, primary_key=True, autoincrement=False),
    Column('value', String(2048), nullable=True),
    Index('type_value', 'type', 'value'),
)

application_table = Table(
    'application',
    metadata,
    Column('application_id', Integer, nullable=False, primary_key=True),
    Column('provider_id', SmallInteger),
    Column('provider_client_id', String(512), nullable=False),
    Column('application_name', String(160), nullable=False),
    Index('application_id', 'application_id', unique=True),
    Index('provider_id', 'provider_id'),
    Index('provider_client_id', 'provider_id', 'provider_client_id', unique=True),
    Index('application_name', 'application_name', unique=True),
    sqlite_autoincrement=True,
)
