# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import (
    date,
    datetime,
)

from sqlalchemy.dialects import sqlite
from sqlalchemy.types import (
    DATE as _DATE,
    TIMESTAMP as _TIMESTAMP,
    TypeDecorator,
    TypeEngine,
)


class TIMESTAMP(TypeDecorator):
    """
    Повторяет для sqlite поведение MySQL в работе со временем
    '0000-00-00 00:00:00'.


    tt = Table(
        Column('name', String(), primary_key=True),
        Column(
            'value', TIMESTAMP,
            server_default='0000-00-00 00:00:00',
            nullable=False,
        ),
    )
    tt.insert().values(name='foo')


    Результаты select([tt])

                            MySQL               sqlite
    sqlalchemy.TIMESTAMP    [('foo', None)]     [('foo', datetime(0, 0, 0, 0, 0))]
    this.TIMESTAMP          [('foo', None)]     [('foo', None)]


    Результаты select([tt]).where(tt.c.value == '0000-00-00 00:00:00')

                            MySQL               sqlite
    sqlalchemy.TIMESTAMP    [('foo', None)]     [('foo', datetime(0, 0, 0, 0, 0))]
    this.TIMESTAMP          [('foo', None)]     [('foo', None)]
    """
    impl = TypeEngine

    def __init__(self, *args, **kwargs):
        super(TIMESTAMP, self).__init__(*args, **kwargs)
        self._args = args
        self._kwargs = kwargs

    def load_dialect_impl(self, dialect):
        if dialect.name == 'sqlite':
            self._kwargs.setdefault(
                'storage_format',
                '%(year)04d-%(month)02d-%(day)02d '
                '%(hour)02d:%(minute)02d:%(second)02d',
            )
            impl = sqlite.DATETIME(*self._args, **self._kwargs)
        else:
            impl = _TIMESTAMP(*self._args, **self._kwargs)
        return impl

    def process_bind_param(self, value, dialect):
        if dialect.name == 'sqlite' and value == str(_zero_datetime):
            value = _zero_datetime
        return value

    def process_result_value(self, value, dialect):
        if dialect.name == 'sqlite' and value == _zero_datetime:
            value = None
        return value


class DATE(TypeDecorator):
    impl = TypeEngine

    def __init__(self, *args, **kwargs):
        super(DATE, self).__init__(*args, **kwargs)
        self._args = args
        self._kwargs = kwargs

    def load_dialect_impl(self, dialect):
        if dialect.name == 'sqlite':
            self._kwargs.setdefault(
                'storage_format',
                '%(year)04d-%(month)02d-%(day)02d',
            )
            impl = sqlite.DATE(*self._args, **self._kwargs)
        else:
            impl = _DATE(*self._args, **self._kwargs)
        return impl

    def process_bind_param(self, value, dialect):
        if dialect.name == 'sqlite' and isinstance(value, basestring):
            value = _SparseDate.from_string(value)
        return value

    def process_result_value(self, value, dialect):
        if dialect.name == 'sqlite' and value == _zero_date:
            value = None
        return value


class _ZeroDatetime(datetime):
    """
    Имитация объекта datetime(0, 0, 0, 0, 0, 0).
    """
    def __new__(cls):
        earliest = datetime.min
        return super(_ZeroDatetime, cls).__new__(
            cls,
            earliest.year,
            earliest.month,
            earliest.day,
        )

    def __str__(self):
        return '0000-00-00 00:00:00'

    def __repr__(self):
        return 'datetime.datetime(0, 0, 0, 0, 0, 0)'

    def replace(self, *args, **kwargs):
        return self

    def __eq__(self, other):
        if not isinstance(other, datetime):
            return NotImplemented
        return (
            self.year == other.year and
            self.month == other.month and
            self.day == other.day and
            self.hour == other.hour and
            self.minute == other.minute and
            self.second == other.second and
            self.microsecond == other.microsecond
        )

    def __ne__(self, other):
        return not self.__eq__(other)

    year = month = day = hour = minute = second = microsecond = property(lambda self: 0)


class _SparseDate(date):
    def __new__(cls, year, month, day):
        if year and month and day:
            return date(year, month, day)

        earliest = date.min
        self = super(_SparseDate, cls).__new__(
            cls,
            year or earliest.year,
            month or earliest.month,
            day or earliest.day,
        )
        self._year = year or 0
        self._month = month or 0
        self._day = day or 0
        return self

    def __str__(self):
        return '%04d-%02d-%02d' % (self.year, self.month, self.day)

    def __repr__(self):
        return 'datetime.date(%s)' % ', '.join(map(str, [self.year, self.month, self.day]))

    def __eq__(self, other):
        if not isinstance(other, date):
            return NotImplemented
        return (
            self.year == other.year and
            self.month == other.month and
            self.day == other.day
        )

    def __ne__(self, other):
        return not self.__eq__(other)

    @property
    def year(self):
        return self._year

    @property
    def month(self):
        return self._month

    @property
    def day(self):
        return self._day

    @classmethod
    def from_string(cls, s):
        if s is None:
            raise ValueError('Invalid date: None')
        bits = s.split('-')
        if len(bits) != 3:
            raise ValueError('Invalid date: %s' % s)
        try:
            bits = map(int, bits)
        except ValueError:
            raise ValueError('Invalid date: %s' % s)
        return cls(*bits)


_zero_datetime = _ZeroDatetime()
_zero_date = _SparseDate(0, 0, 0)
