# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.common.misc import trim_message
from sqlalchemy import sql
from sqlalchemy.engine import create_engine as sqlalchemy_create_engine


logger = logging.getLogger(__name__)

_CONFIG_ATTRS = [
    'host',
    'database',
    'user',
    'password',
    'port',
    'connect_timeout',
    'read_timeout',
    'reconnect_retries',
    'write_timeout',
    'pool_size',
    'pool_timeout',
]


# О значение параметров читай на
# https://dev.mysql.com/doc/refman/5.7/en/mysql-options.html
# https://github.com/PyMySQL/mysqlclient-python/blob/master/doc/user_guide.rst#functions-and-attributes
_MYSQLDB_CONNECT_ARGS = [
    'connect_timeout',
    'read_timeout',
    'write_timeout',
]

_YANDEX_MYSQLCONNECTOR_CONNECT_ARGS = [
    'connect_timeout',
    # read_timeout и write_timeout есть только в яндексовой версии
    # mysql-connector-python
    'read_timeout',
    'write_timeout',
]


def build_master_db_config(config):
    db_config = dict()
    for attr in _CONFIG_ATTRS:
        db_config[attr] = getattr(config, 'db_master_' + attr)
    return db_config


def build_slave_db_config(config):
    db_config = dict()
    for attr in _CONFIG_ATTRS:
        db_config[attr] = getattr(config, 'db_slave_' + attr)
    return db_config


def create_engine(*args, **kwargs):
    return mysql_client_create_engine(*args, **kwargs)


def mysql_client_create_engine(db_config, echo=True, pool_recycle=3600):
    logger.debug('Creating DB engine')

    dsn = '%s://%s:%s@%s:%s/%s' % (
        'mysql',
        db_config['user'],
        db_config['password'],
        db_config['host'],
        db_config['port'],
        db_config['database'],
    )

    connect_args = dict(
        charset='utf8mb4',
        init_command='set names "utf8mb4"',
    )
    for key in _MYSQLDB_CONNECT_ARGS:
        if key in db_config:
            connect_args[key] = db_config[key]

    # Включаем честный autocommit в MySQL, и выключаем его имитацию
    # в Sqlalchemy через пару оператор + commit. Имитация ведёт к проблемам,
    # когда соединение теряется после выполнения оператора и до выполнения
    # commit.
    connect_args.update(autocommit='true')
    execution_options = dict(autocommit=False)

    engine = sqlalchemy_create_engine(
        dsn,
        connect_args=connect_args,
        execution_options=execution_options,
        echo=echo,
        # Внимание: pool_size и max_overflow работают в паре. В документации к
        # sqlalchemy их значение описано с ошибками. Обязательно проверяейте
        # стратегию работы с пулом после внесения изменений в эти параметры.
        pool_size=db_config['pool_size'],
        max_overflow=0,
        pool_timeout=db_config['pool_timeout'],
        pool_recycle=pool_recycle,
    )

    engine.reconnect_retries = db_config['reconnect_retries']

    return engine


def mysql_connector_create_engine(db_config, echo=True, pool_recycle=3600):
    logger.debug('Creating DB engine')

    dsn = '%s://%s:%s@%s:%s/%s' % (
        'mysql+mysqlconnector',
        db_config['user'],
        db_config['password'],
        db_config['host'],
        db_config['port'],
        db_config['database'],
    )

    connect_args = dict(
        charset='utf8mb4',
        # Под gevent следует запускать python-версию, т.к. gevent не умеет
        # манкипатчить c-extensions.
        use_pure=True,
    )
    for key in _YANDEX_MYSQLCONNECTOR_CONNECT_ARGS:
        if key in db_config:
            connect_args[key] = db_config[key]

    engine = sqlalchemy_create_engine(
        dsn,
        connect_args=connect_args,
        echo=echo,
        # Внимание: pool_size и max_overflow работают в паре. В документации к
        # sqlalchemy их значение описано с ошибками. Обязательно проверяейте
        # стратегию работы с пулом после внесения изменений в эти параметры.
        pool_size=db_config['pool_size'],
        max_overflow=0,
        pool_timeout=db_config['pool_timeout'],
        pool_recycle=pool_recycle,
    )

    engine.reconnect_retries = db_config['reconnect_retries']

    return engine


def get_master_engine():
    return LazyLoader.get_instance('master_db_engine')


def get_slave_engine():
    return LazyLoader.get_instance('slave_db_engine')


def database_query_to_str(query, engine=None):
    if not engine:
        engine = get_slave_engine()
    compiled = query.compile(
        engine,
        compile_kwargs={'literal_binds': True},
    )
    return trim_message(compiled.__str__(), cut=False)


def is_read_only_db_query(query):
    return not isinstance(query, (sql.Update, sql.Delete, sql.Insert))
