# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re
import sys


class CauseExceptionMixin(object):
    def setup_cause(self):
        if not getattr(self, 'cause', None):
            exc_type, exc = sys.exc_info()[:2]
            if exc_type is not None and issubclass(exc_type, object):
                self.cause = exc


class SocialException(CauseExceptionMixin, Exception):
    def __init__(self, *more):
        if not hasattr(self, 'code'):
            class_name = self.__class__.__name__
            match = re.search(r'^([A-Z][a-z]+)([A-Z][a-z]+)$', class_name)
            if match:
                self.code = "%s-%s" % (match.group(1).lower(), match.group(2).lower())

        super(SocialException, self).__init__(*more)

        self.setup_cause()


class FailureSourceType(object):
    network = 'network'
    database = 'database'

    # External относится к отказам соц. провайдеров (временный отказ,
    # неожиданный ответ и т.д.) Обычно это означает что у провайдеры что-то
    # сломалось.
    external = 'external'

    # Not_error означает что отказ не требует починки, например (пользователь
    # отказался авторизовать приложение, у потребителя нет грантов на вызов
    # ручки, Социализм потротлили из-за слишком частых запросов).
    not_error = 'not_error'

    # Internal относится ко всему что не попадает в остальные группы.
    internal = 'internal'


class NotErrorFailureSourceTypeMixin(object):
    failure_source_type = FailureSourceType.not_error


class DatabaseFailureSourceTypeMixin(object):
    failure_source_type = FailureSourceType.database


class ExternalFailureSourceTypeMixin(object):
    failure_source_type = FailureSourceType.external


class NetworkFailureSourceTypeMixin(object):
    failure_source_type = FailureSourceType.network


class ProviderUnknown(SocialException):
    pass


class ApplicationUnknown(SocialException):
    pass


class UnrefreshableTokenError(SocialException):
    pass


class GrantsMissingError(SocialException):
    def __init__(self, description, grants_context, missing_grants=None,
                 failed_to_parse_ticket=False):
        super(GrantsMissingError, self).__init__(description)
        self.grants_context = grants_context
        self.missing_grants = missing_grants
        self.failed_to_parse_ticket = failed_to_parse_ticket


class GrantsContextError(SocialException):
    pass


class TvmTicketParsingGrantsContextError(GrantsContextError):
    pass


class RateLimitExceededError(SocialException):
    pass


class OfflineZoraUseragentNotImplementedError(SocialException):
    pass


class DatabaseError(DatabaseFailureSourceTypeMixin, SocialException):
    pass


class NotFound(NotErrorFailureSourceTypeMixin, SocialException):
    code = 'not_found'


class BasicProxylibError(CauseExceptionMixin, Exception):
    # Является ли это исключение чем-то необычным, на что надо обратить внимание,
    # либо это обычная ошибка вроде недостаточных прав или отсутствия токена у пользователя.
    is_normal_workflow_error = False

    def __init__(self, *args):
        super(BasicProxylibError, self).__init__(*args)
        self.setup_cause()

    def __str__(self):
        # Чтобы не появлялось глупых сообщений "None", когда используется str(e).
        if self.args == (None,):
            return ''
        return super(BasicProxylibError, self).__str__()


class InternalProxylibError(BasicProxylibError):
    pass


class ProviderUnknownProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class ProviderCommunicationProxylibError(ExternalFailureSourceTypeMixin, BasicProxylibError):
    def __init__(self, message=None, description=None):
        if message:
            args = [message]
        else:
            args = list()
        super(ProviderCommunicationProxylibError, self).__init__(*args)

        self.message = message
        self.description = description

    def __copy__(self):
        cls = type(self)
        return cls(self.message, self.description)


class NetworkProxylibError(NetworkFailureSourceTypeMixin, ProviderCommunicationProxylibError):
    pass


class ProviderTemporaryUnavailableProxylibError(ProviderCommunicationProxylibError):
    """
    Произошёл временный отказ одной из внешних для Яндекса систем.
    """


class UnexpectedResponseProxylibError(ProviderCommunicationProxylibError):
    pass


class TooLongUnexpectedResponseProxylibError(UnexpectedResponseProxylibError):
    def __init__(self, description, response_length, max_response_length):
        super(TooLongUnexpectedResponseProxylibError, self).__init__(description=description)
        self.response_length = response_length
        self.max_response_length = max_response_length

    def __copy__(self):
        cls = type(self)
        return cls(self.description, self.response_length, self.max_response_length)


class InvalidResponseAttributeProxylibError(UnexpectedResponseProxylibError):
    def __init__(
        self,
        description,
        attr_name,
        max_length=None,
        min_length=None,
    ):
        super(InvalidResponseAttributeProxylibError, self).__init__(description=description)
        self.attr_name = attr_name
        self.max_length = max_length
        self.min_length = min_length

    def __copy__(self):
        cls = type(self)
        return cls(self.description, self.attr_name, self.max_length, self.min_length)


class ResponseDecodeProxylibError(UnexpectedResponseProxylibError):
    pass


class ProviderRateLimitExceededProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class SocialUserDisabledProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class CaptchaNeededProxylibError(BasicProxylibError):
    def __init__(self, sid=None, url=None):
        super(CaptchaNeededProxylibError, self).__init__()
        self.sid = sid
        self.url = url

    is_normal_workflow_error = True


class ValidationRequiredProxylibError(BasicProxylibError):
    def __init__(self, validation_url=None):
        super(ValidationRequiredProxylibError, self).__init__()
        self.validation_url = validation_url

    is_normal_workflow_error = True


class AlbumNotExistsProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class InvalidTokenProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class PermissionProxylibError(BasicProxylibError):
    is_normal_workflow_error = True


class BadParametersProxylibError(BasicProxylibError):
    """В прокси пришли с неверными параметрами"""
    is_normal_workflow_error = True
