# -*- coding: utf-8 -*-

from functools import wraps
import logging
import os
import sys

from passport.backend.core.dynamic_config import LoadConfigsError
from passport.backend.core.grants.grants_config import GrantsConfig as BaseGrantsConfig
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.common.chrono import now
from passport.backend.social.common.exception import (
    GrantsMissingError,
    TvmTicketParsingGrantsContextError,
)
from ticket_parser2.exceptions import TicketParsingException
import yenv


logger = logging.getLogger(__name__)


def find_all_consumers_by_context(grants_config, context):
    if context.ticket_body:
        try:
            context.load_consumer_from_tvm_ticket(grants_config)
        except TicketParsingException as e:
            logger.debug(
                'Failed to parse TVM ticket for %s: %s (%s)' % (
                    context.to_str(with_ticket_body=True),
                    e.message,
                    e.debug_info,
                ),
            )
            raise TvmTicketParsingGrantsContextError()
    if context.consumer_from_tvm:
        consumers = [context.consumer_from_tvm]
    elif context.consumer:
        if grants_config.is_valid_ip(context.consumer_ip, context.consumer):
            consumers = [context.consumer]
            consumers = grants_config.filter_out_consumers_with_tvm_client(consumers)
        else:
            consumers = list()
    else:
        consumers = grants_config.get_consumers_by_ip(context.consumer_ip)
        consumers = grants_config.filter_out_consumers_with_tvm_client(consumers)
    return consumers


class GrantsConfig(BaseGrantsConfig):
    _instances = {}

    def __new__(cls, *args, **kwargs):
        project_name = args[0]
        instance = cls._instances.get(project_name)
        if instance is None:
            cls._instances[project_name] = super(GrantsConfig, cls).__new__(cls)
        return cls._instances[project_name]

    def __init__(
        self,
        project_name,
        tvm_credentials_manager,
        backup_tvm_credentials_manager=None,
    ):
        self.project_name = project_name
        self._tvm_credentials_manager = tvm_credentials_manager
        self._backup_tvm_credentials_manager = backup_tvm_credentials_manager
        super(GrantsConfig, self).__init__(
            grants_paths=self._make_grants_paths(),
        )
        self.load()

    def _make_grants_paths(self):
        # Это место мокается в тестах, поэтому для прогона тестов не нужен
        # social_grants.
        import social_grants
        config_dir_path = os.path.abspath(os.path.dirname(social_grants.__file__))
        env = 'production' if yenv.type == 'rc' else yenv.type
        return [
            '%s/%s/%s.json' % (config_dir_path, env, self.project_name),
        ]

    def merge_configs(self, configs):
        raise NotImplementedError('Not used')  # pragma: no cover

    def postprocess(self):
        new_config = {}
        for key in self.config.keys():
            grants = self.config[key].get('grants', [])
            new_config[key] = {
                'networks': self.config[key]['networks'],
                'grants': {grant: '*' for grant in grants},
            }

            tvm_client_config = self.config[key].get('client', dict())
            if tvm_client_config:
                new_config[key]['client'] = tvm_client_config

        self.config = new_config

        super(GrantsConfig, self).postprocess()

        self._tvm_client_id_to_consumer = {client_id: consumer for consumer, client_id in self._tvm_client_id_cache.iteritems()}

    def load(self):
        # Это починенная версия passport.backend.core.BaseDynamicConfig.load,
        # в ней expires_at обновляется даже, когда конфигурационных файл не
        # изменился.

        if not self.is_expired:
            return

        try:
            new_config = self._read_configs()
        except LoadConfigsError as e:
            if not self.config:
                logger.error('Could not load initial %s from file with error: %s', self.__class__.__name__, e)
                raise

            logger.warning('Using old %s, could not load from file with error: %s', self.__class__.__name__, e)
            return

        self.expires_at = now.f() + self.random_cache_time()

        # Изменений в конфигах не было, поэтому нам нечего делать
        if new_config is None:
            return
        self.config = new_config

        logger.info('Using new %s' % self.__class__.__name__)
        self.postprocess()

        # Эта часть сделана только в Социализме
        if self._tvm_credentials_manager is not None:
            self._tvm_credentials_manager.load()
        if self._backup_tvm_credentials_manager is not None:
            self._backup_tvm_credentials_manager.load()

    def check_tvm_ticket(self, ticket_body):
        try:
            return self._tvm_credentials_manager.service_context.check(ticket_body)
        except TicketParsingException:
            if self._backup_tvm_credentials_manager is None:
                raise

            first_exc_info = sys.exc_info()
            try:
                ticket = self._backup_tvm_credentials_manager.service_context.check(ticket_body)
                logger.debug('TVM service ticket is issued for backup TVM client')
                return ticket
            except TicketParsingException:
                raise first_exc_info[0], first_exc_info[1], first_exc_info[2]

    def find_consumer_by_tvm_client_id(self, client_id):
        return self._tvm_client_id_to_consumer.get(client_id)

    def filter_out_consumers_with_tvm_client(self, consumers):
        return [c for c in consumers if c not in self._tvm_client_id_cache]

    def get_consumers_by_ip(self, ip):
        return self._get_consumers_by_ip(ip)


class Grants(object):
    """
    Объект, который используется как декоратор.

    Пример:

    def get_context():
        return build_grants_context()

    def access_denied(reason):
        return dict(
            code=403,
            name='access-denied',
            description='Access denied (%s)' % str(reason),
        )

    grants = Grants(grants_config, context_getter=context_getter, error_func=access_denied)
    """
    def __init__(self, context_getter, error_func):
        """
        :param config: grants config
        :type config: GrantsConfig
        :param context_getter: callable для получения контекста запроса
        :param error_func: callable, вызываемый в случае отсутствия грантов
        """
        self.context_getter = context_getter
        self.error_func = error_func

    def __call__(self, grants, additional_grant_func=None):
        def wrapper(f):
            @wraps(f)
            def _wrapper(*args, **kwargs):
                return self._check_grants_and_call_function(
                    f,
                    wrapper,
                    additional_grant_func,
                    args,
                    kwargs,
                )

            return _wrapper

        if not hasattr(grants, '__iter__'):
            grants = [grants]

        wrapper.grants = grants

        return wrapper

    def _check_grants_and_call_function(self, ok_func, wrapper, additional_grant_func, args, kwargs):
        required_grants = wrapper.grants[:]
        if callable(additional_grant_func):
            required_grants.extend(additional_grant_func(*args, **kwargs))

        try:
            self.check_any_of_grants(required_grants)
        except GrantsMissingError as e:
            return self.error_func(e)

        return ok_func(*args, **kwargs)

    def check_any_of_grants(self, grants):
        """
        Проверить наличие грантов.
        """
        context = self.context_getter()

        try:
            allowed_grants = check_any_of_grants(get_grants_config(), context, grants)
        except GrantsMissingError as e:
            logger.warning('Access denied (%s)' % str(e))
            raise

        logger.info('Access to %s is granted for %s' % (sorted(allowed_grants), str(context)))
        return allowed_grants


class GrantsContext(object):
    def __init__(self, consumer_ip, consumer=None, ticket_body=None):
        self.consumer_ip = consumer_ip
        self.consumer = consumer
        self.ticket_body = ticket_body
        self.tvm_client_id = None
        self.consumer_from_tvm = None
        self.matching_consumers = None  # потребители из конфига грантов, подходящие под контекст

    def _truncated_ticket_body(self):
        ticket_body = str(self.ticket_body)
        if len(ticket_body) > 12:
            ticket_body = ticket_body[:12] + '...'
        return ticket_body

    def __str__(self):
        return self.to_str()

    def to_str(self, with_ticket_body=False):
        bits = []

        if self.consumer_ip:
            consumer_ip = 'ip = %s' % self.consumer_ip
            bits.append(consumer_ip)

        if self.consumer:
            consumer = 'name = %s' % self.consumer
            bits.append(consumer)

        if self.matching_consumers:
            matching_consumers = 'matching_consumers = %s' % (';'.join(sorted(self.matching_consumers)))
            bits.append(matching_consumers)

        if self.consumer_from_tvm:
            consumer_from_tvm = 'name_from_tvm = %s' % self.consumer_from_tvm
            bits.append(consumer_from_tvm)

        if self.tvm_client_id:
            bits.append('tvm_client_id = %s' % self.tvm_client_id)

        if not self.consumer_from_tvm and self.ticket_body and with_ticket_body:
            bits.append('ticket = "%s"' % self._truncated_ticket_body())

        return 'Consumer(%s)' % ', '.join(bits)

    def load_consumer_from_tvm_ticket(self, grants_config):
        if self.consumer_from_tvm is not None:
            return
        ticket = grants_config.check_tvm_ticket(self.ticket_body)
        self.tvm_client_id = ticket.src
        self.consumer_from_tvm = grants_config.find_consumer_by_tvm_client_id(ticket.src)


class _GrantsChecker(object):
    def check(self, grants_config, context, required_grants):
        consumers = self._find_all_consumers_by_context(grants_config, context)

        required_grants = frozenset(required_grants)
        for consumer in consumers:
            perms = self._get_permissions(
                grants_config,
                context,
                consumer,
                required_grants=required_grants,
            )
            if self._is_access_granted(required_grants, perms.missing_required):
                return required_grants - perms.missing_required

        comma_separated_grants = ', '.join(required_grants)
        raise GrantsMissingError(
            'Grants %s are missing for %s' % (comma_separated_grants, str(context)),
            grants_context=context,
            missing_grants=required_grants,
        )

    def filter_allowed_grants(self, grants_config, context, grants):
        """
        Находит какие из grants есть у потребителя
        """
        consumers = self._find_all_consumers_by_context(grants_config, context)

        if not consumers:
            return set()

        grants = set(grants)
        least_missing_grants = grants
        for consumer in consumers:
            perms = self._get_permissions(
                grants_config,
                context,
                consumer,
                optional_grants=grants,
            )
            if len(perms.missing_optional) < len(least_missing_grants):
                least_missing_grants = perms.missing_optional
            if not least_missing_grants:
                break
        return grants - least_missing_grants

    def _find_all_consumers_by_context(self, grants_config, context):
        try:
            consumers = find_all_consumers_by_context(grants_config, context)
            context.matching_consumers = consumers
            return consumers
        except TvmTicketParsingGrantsContextError:
            raise GrantsMissingError('Failed to parse TVM ticket', grants_context=context, failed_to_parse_ticket=True)

    def _get_permissions(
        self,
        grants_config,
        context,
        consumer,
        required_grants=frozenset(),
        optional_grants=frozenset(),
    ):
        return grants_config.is_permitted(
            required_grants=required_grants,
            optional_grants=optional_grants,
            ip=context.consumer_ip,
            consumer=consumer,
            tvm_client_id=context.tvm_client_id,
        )

    def _is_access_granted(self, required_grants, missing_grants):
        raise NotImplementedError()  # pragma: no cover


class _AllGrantsChecker(_GrantsChecker):
    def _is_access_granted(self, required_grants, missing_grants):
        return len(missing_grants) == 0


class _AnyGrantsChecker(_GrantsChecker):
    """
    Требует наличия хотя бы одного гранта из переданных.
    """
    def _is_access_granted(self, required_grants, missing_grants):
        return required_grants > missing_grants


check_all_of_grants = _AllGrantsChecker().check
check_any_of_grants = _AnyGrantsChecker().check
filter_allowed_grants = _GrantsChecker().filter_allowed_grants


def get_grants_config():
    return LazyLoader.get_instance('grants_config')
