# -*- coding: utf-8 -*-
"""
В этом модуле выполняется gevent-манкипатчинг, поэтому нужно быть предельно
осторожным в его написании и не импортировать другие модули до манкипатчинга.
"""

from __future__ import absolute_import

import sys

import gevent.socket
import gunicorn.app.base
import gunicorn.app.wsgiapp
import gunicorn.util
import gunicorn.workers.ggevent


def main():
    gunicorn.app.wsgiapp.run()


def setup_worker():
    from gevent import monkey
    monkey.noisy = False
    monkey.patch_all()

    import greenify
    greenify.greenify()


class _WsgiServer(gunicorn.workers.ggevent.PyWSGIServer):
    # Благодаря этому gevent будет давать больший приоритет уже выполняемым
    # запросам, вместо того чтобы брать всё больше новых запросов на обработку.
    # Как следствие:
    #   * запросы будут выполнятся настолько быстро насколько возможно
    #   * запросы будут честней распределятся между процессами
    max_accept = 1


class GeventWorker(gunicorn.workers.ggevent.GeventPyWSGIWorker):
    server_class = _WsgiServer

    def patch(self):
        self.log.info('Monkeys are patching environment')

        setup_worker()

        # monkey patch sendfile to make it none blocking
        gunicorn.workers.ggevent.patch_sendfile()

        # patch sockets
        sockets = []
        for s in self.sockets:
            if sys.version_info[0] == 3:
                sockets.append(
                    gevent.socket.socket(
                        s.FAMILY,
                        gevent.socket.SOCK_STREAM,
                        fileno=s.sock.fileno(),
                    ),
                )
            else:
                sockets.append(
                    gevent.socket.socket(
                        s.FAMILY,
                        gevent.socket.SOCK_STREAM,
                        _sock=s,
                    ),
                )
        self.sockets = sockets


class Application(gunicorn.app.base.BaseApplication):
    def __init__(self, cfg):
        self.gapp_cfg = cfg
        super(Application, self).__init__()

    def load(self):
        return gunicorn.util.import_app(self.gapp_cfg['app_uri'])

    def load_config(self):
        for key in self.gapp_cfg:
            if key != 'app_uri':
                self.cfg.set(key, self.gapp_cfg[key])

    def run(self):
        if self.cfg.daemon:
            gunicorn.util.daemonize(self.cfg.enable_stdio_inheritance)
        super(Application, self).run()


if __name__ == '__main__':
    main()
