# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from time import (
    localtime,
    strftime,
)

from flask import request
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.core.logging_utils.loggers.access import AccessLogger
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.misc import (
    GraphiteMessageType,
    name_for_graph_log,
    request_context_to_application_info,
    write_graph_log_message,
)


logger = logging.getLogger(__name__)


_ENDPOINT_TO_HANDLER_ID = dict(
    AuthzInAppForTokenCallbackHandler='authz_in_app.callback',
    AuthzInAppForTokenContinueHandler='authz_in_app.continue',
    AuthzInAppForTokenEntrustToAccountHandler='authz_in_app.entrust_to_account',
    AuthzInAppForTokenStartHandler='authz_in_app.start',
    BindByTokenHandler='bind_by_token',
    BindHandler='bind',
    CallbackHandler='callback',
    CheckPkceHandler='check_pkce',
    ContinueHandler='continue',
    PingHandler='ping',
    RedirectHandler='redirect',
    RetryHandler='retry',
    StartHandler='start',
    TaskByTokenHandler='task_by_token',
    TwitterReverseAuthTokenHandler='tw_reverse_auth_token',
)


def setup_request():
    request_ctx.clear()
    request_ctx.request_id = request.id
    if request.endpoint in _ENDPOINT_TO_HANDLER_ID:
        request_ctx.handler_id = _ENDPOINT_TO_HANDLER_ID[request.endpoint]
    else:
        request_ctx.handler_id = request.endpoint
    request.started_at = now.f()


def log_http_request():
    logger.debug('Dispatching the request (%s)' % request.path)


def log_api_call():
    handler_id = getattr(request_ctx, 'handler_id', None)
    if handler_id:
        logger.debug(
            'Spawning a new handler %s, args=%s, form=%s' % (
                handler_id,
                mask_sensitive_fields(request.args),
                mask_sensitive_fields(request.form),
            ),
        )


def after_request(tskv_format, response):
    handler_id = getattr(request_ctx, 'handler_id', None)
    request_id = getattr(request_ctx, 'request_id', None)
    api_status = getattr(response, 'api_status', None)
    api_error_code = getattr(response, 'api_error_code', None)

    app_info = request_context_to_application_info(request_ctx)

    unixtime = '%.3f' % request.started_at
    timestamp = strftime('%d/%b/%Y:%H:%M:%S', localtime(request.started_at))

    duration = now.f() - request.started_at

    AccessLogger().log(
        tskv_format=tskv_format,
        method=request.method,
        url=request.path,
        handle_id=handler_id,
        request_id=request_id,
        provider=app_info['provider_code'],
        application=app_info['application_name'],
        status=response.status_code,
        api_status=api_status,
        api_error_code=api_error_code,
        unixtime=unixtime,
        timestamp=timestamp,
        duration='%.3f' % duration,
        consumer_ip=request.consumer_ip,
    )

    if handler_id:
        app = getattr(request_ctx, 'application', None)
        app_code = name_for_graph_log(app)
        if not app_code:
            provider = getattr(request_ctx, 'provider', None)
            try:
                app_code = provider['code']
            except (KeyError, TypeError):
                app_code = None
        write_graph_log_message(GraphiteMessageType.request, handler_id, app_code)

    if not handler_id:
        handler_id = request.path

    logger.debug(
        'Response: handler=%s, status=%s, runtime=%.3f, data=%s' %
        (handler_id, response.status_code, duration, response.data),
    )

    request_ctx.clear()

    return response
