# -*- coding: utf-8 -*-

from copy import copy as _copy
from importlib import import_module
import json
import logging
from os import getpid
from pkgutil import walk_packages
import re
import traceback
from urllib import (
    quote as _urllib_quote,
    urlencode as _urlencode,
)
from urlparse import (
    parse_qs as _urlparse_qs,
    parse_qsl as _urlparse_qsl,
)
from uuid import uuid4

import gevent.monkey
from passport.backend.core import Undefined
from passport.backend.core.types.social_business_info import BusinessInfo
from passport.backend.social.common.chrono import (
    datetime_to_unixtime,
    datetime_to_unixtimef,
    now,
)
from passport.backend.utils.common import noneless_dict
from passport.backend.utils.string import smart_text


USERID_TYPE_BUSINESS = 0
USERID_TYPE_SIMPLE = 1

X_TOKEN_SCOPE = 'oauth:grant_xtoken'

# Для универсальности и поддержки потенциальных чужих "Бизнесов" (маловероятно, но все же)
# у нас есть business_id в таблице. Всегда используем единицу.
FACEBOOK_BUSINESS_ID = 1

PLACE_FRAGMENT = 'fragment'
PLACE_QUERY = 'query'

logger = logging.getLogger('social_common')

re_scope = re.compile('[ ,]+')
re_valuable_access = re.compile('valuable access', re.IGNORECASE)
re_avatar_key = re.compile(r'^(\d+)x(\d+)$', re.IGNORECASE)

SCOPE_DELIMITER = ','


def split_scope_string(scope_string):
    if not scope_string:
        return []

    scope = scope_string.strip()
    if scope:
        scopes = re_valuable_access.sub('VALUABLE_ACCESS', scope)
        scopes = re_scope.split(scopes)
    else:
        return []

    scopes = filter(bool, scopes)
    return scopes


def build_scope_string(scopes, delimiter=SCOPE_DELIMITER):
    return delimiter.join(scopes)


def rebuild_scope_string(scope_string, delimiter):
    scopes = split_scope_string(scope_string)
    return delimiter.join(scopes)


def write_graph_log_message(*args):
    """
    Формат:
    [GRAPH_LOG=requests.<method>.<provider>]
    [GRAPH_LOG=errors.<method>.<provider>.(internal|network)]
    [GRAPH_LOG=warnings.<provider>]
    [GRAPH_LOG=other.<provider>]
    """
    sargs = list()
    for arg in args:
        if arg is None:
            arg = '-'
        elif not isinstance(arg, basestring):
            arg = str(arg)
        if not arg:
            arg = '-'
        sargs.append(arg)
    sargs = [s.replace('.', '-') for s in sargs]
    msg = '.'.join(sargs)
    msg = '[GRAPH_LOG=%s]' % msg
    logger.info(msg)


def name_for_graph_log(application):
    if application and application.provider:
        return application.provider['code']
    if application:
        return application.name


def request_context_to_application_info(request_context):
    application = getattr(request_context, 'application', None)

    provider_code = None
    if application and application.provider:
        provider_code = application.provider['code']
    if not provider_code:
        provider = getattr(request_context, 'provider', None)
        if provider:
            provider_code = provider['code']

    app_name = None
    if application:
        app_name = application.name

    return dict(
        provider_code=provider_code,
        application_name=app_name,
    )


def trim_message(message, cut=True):
    message = smart_text(message, errors='replace')

    if cut and len(message) > 1024:
        message = "%s ... (%d total symbols)" % (message[:1024], len(message))

    return message.replace('\n', r'\n').replace('\r', r'\r')


def format_traceback():
    return traceback.format_exc().replace('\n', r'\n')


def get_max_size_avatar(avatars):
    if not avatars or not isinstance(avatars, dict):
        logger.debug('No avatars found.')
        return

    # Получая словарь вида {'50x0': url1, '100x200': url2, '0x0': url3...},
    # вернем ссылку на картинку с наибольшим разрешением. Точные размеры без выкачивания фоток
    # мы не знаем, поэтому это только эвристика. Однако, не угадать с размером и загрузить
    # маленькую аватарку - не смертельно.
    max_key = None
    max_size = -1

    for key in avatars:
        result = re_avatar_key.match(key)
        if not result:
            continue
        size = sorted(map(int, result.group(1, 2)), reverse=True)
        if size > max_size:
            max_size = size
            max_key = key

    if not max_key:
        logger.debug('Failed to detect a url to upload.')
        return

    return avatars[max_key]


# TODO
# Завести самостоятельный объект userid

def get_business_userid(business_id, business_token):
    business_info = BusinessInfo(id=business_id, token=business_token)
    return business_info.to_userid()


def parse_userid(userid):
    if not userid.startswith('bt:'):
        return USERID_TYPE_SIMPLE, userid

    items = userid.split(':', 2)
    if len(items) != 3:
        raise ValueError('Invalid userid %s' % userid)

    return USERID_TYPE_BUSINESS, (int(items[1]), items[2])


def is_simple_userid(userid):
    _type, _ = parse_userid(userid)
    return _type is USERID_TYPE_SIMPLE


def build_request_id():
    return '%d-%s-%d' % (getpid(), uuid4().hex, datetime_to_unixtimef(now()))


def expires_in_to_expires_at(e_time):
    """
    0 и None означают, что сущность вечная и возвращается None, другая дельта
    выводится в момент времени в будущем.
    """
    if e_time is not None:
        e_time = int(e_time)
    if e_time:
        e_time = get_future_timestamp(e_time)
    else:
        e_time = None
    return e_time


def get_future_timestamp(seconds):
    return datetime_to_unixtime(now()) + int(seconds)


def build_dict_from_standard(standard=None, values=None, exclude_attrs=None):
    """
    Строит словарь по заданному эталону заменяя или удаляя переданные ключи.
    """
    if standard is not None:
        retval = standard.copy()
    else:
        retval = dict()
    retval.update(values or dict())
    if exclude_attrs is None:
        exclude_attrs = []
    for attr in exclude_attrs:
        del retval[attr]
    return retval


def build_dict_list_from_standard(standard, args_list):
    retval = []
    for args in args_list:
        values = args.get('values', {})
        exclude_attrs = args.get('exclude_attrs')
        retval.append(build_dict_from_standard(standard, values, exclude_attrs))
    return retval


def urlparse_qsl(query, keep_blank_values=False):
    if isinstance(query, unicode):
        # _urlparse_qsl неправильно расквотирует параметры из юникодных
        # строк, поэтому нужно закодировать.
        query = query.encode('utf-8')
    return _urlparse_qsl(query, keep_blank_values)


def urlparse_qs(query, keep_blank_values=False):
    if isinstance(query, unicode):
        # _urlparse_qs неправильно расквотирует параметры из юникодных
        # строк, поэтому нужно закодировать.
        query = query.encode('utf-8')
    return _urlparse_qs(query, keep_blank_values)


def urlencode(args):
    # _urlencode не умеет работать с юникодными значениями, так что перед его
    # использованием их нужно закодировать.
    args = _encode_fields(args)
    return _urlencode(args)


def urllib_quote(string, safe=None):
    if isinstance(string, unicode):
        string = string.encode('utf-8')
    optional_args = []
    if safe is not None:
        optional_args.append(safe)
    return _urllib_quote(string, *optional_args)


def is_account_like_portalish(account):
    return (
        account.is_neophonish or
        account.is_normal or
        account.is_social or
        account.is_lite or
        account.is_pdd
    )


def _encode_fields(fields):
    if not fields:
        return fields

    if hasattr(fields, 'items'):
        fields_is_dict = True
        fields = fields.items()
    else:
        fields_is_dict = False

    encoded_fields = []
    for key, value in fields:
        if isinstance(key, unicode):
            key = key.encode('utf-8')
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        encoded_fields.append((key, value))

    if fields_is_dict:
        encoded_fields = dict(encoded_fields)

    return encoded_fields


def dump_to_json_string(obj, minimal=False):
    if minimal:
        return _SocialJsonEncoder(separators=(',', ':')).encode(obj)
    else:
        return _SocialJsonEncoder(indent=4).encode(obj) + '\n'


def remove_undefined_values(d):
    if hasattr(d, 'iteritems'):
        retval = {k: v for k, v in d.iteritems() if v is not Undefined}
    else:
        retval = [(k, v) for k, v in d if v is not Undefined]
    return retval


def discover_modules(package):
    modules = []
    for _, name, is_pkg in walk_packages(package.__path__):
        if not is_pkg:
            modules.append(
                import_module(
                    '.' + name,
                    package.__name__,
                ),
            )
    return modules


def discover_subclasses(module, superclass):
    module_attrs = [getattr(module, n) for n in dir(module)]
    subclasses = []
    for attr in module_attrs:
        try:
            if attr is not superclass and issubclass(attr, superclass):
                subclasses.append(attr)
        except TypeError:
            pass
    return subclasses


def in_gevent():
    return gevent.monkey.is_module_patched('socket')


def failsafe(func, *args, **kwargs):
    try:
        retval = func(*args, **kwargs)
        return retval, None
    except Exception as e:
        logger.debug('Failsafe call raised exception', exc_info=True)
        return None, e


def copy(obj, fallback=None):
    try:
        return _copy(obj)
    except Exception:
        if fallback is None:
            raise
        logger.error('Failed to copy %r, fallback to %r' % (obj, fallback), exc_info=True)
        return obj


class GraphiteMessageType(object):
    error = 'errors'
    request = 'requests'
    other = 'other'


class ApplicationMapper(object):
    """
    Отображение пары (код соц. провайдера, приложение) в соответствующий объект
    или специальный объект (default).
    """
    def __init__(self):
        self._defaults = {}
        self._mapping = {}

    def add_mapping(self, provider_code, default, mapping=None):
        """
        Отображать приложения для провайдера provider_code в default, кроме тех
        приложений, движки которых описаны в mapping.
        """
        mapping = mapping or {}
        self._defaults[provider_code] = default
        for engine_name, value in mapping.iteritems():
            self._mapping[(provider_code, engine_name)] = value

    def map_application(self, provider_code, application):
        default = self._defaults[provider_code]
        if not application:
            return default
        key = (provider_code, application.engine_id)
        return self._mapping.get(key, default)

    def has_provider(self, provider_code):
        return provider_code in self._defaults

    def __getitem__(self, key):
        return self.map_application(*key)


class _SocialJsonEncoder(json.JSONEncoder):
    def __init__(
        self,
        indent=None,
        separators=None,
    ):
        super(_SocialJsonEncoder, self).__init__(
            indent=indent,
            separators=separators,
            sort_keys=True,
        )


class class_property(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, obj, type):
        return self.func(type)


class UserParamDescriptor(object):
    """
    Описание параметра, который передаётся в ручки Социализма потребителем, а
    затем пробрасывается в ручки провайдера.
    """

    TYPE_DATA = 'data'
    TYPE_HEADER = 'header'

    def __init__(
        self,
        name=None,
        type=None,
    ):
        self.name = name
        self.type = type

    @classmethod
    def from_str(cls, s):
        data = json.loads(s)
        return cls(name=data.get('name'), type=data.get('type'))

    def __str__(self):
        return dump_to_json_string(noneless_dict(dict(name=self.name, type=self.type)))
