# -*- coding: utf-8 -*-

import json
import re


class SlotObject(object):
    __slots__ = ()  # необходимо переопределить в потомках

    def __init__(self, **kwargs):
        for attr in self.__slots__:
            value = kwargs.get(attr)
            setattr(self, attr, value)

    def __str__(self):
        return self.__unicode__()

    def __unicode__(self):
        def getattr_unicode(attr):
            val = getattr(self, attr)
            if isinstance(val, unicode):
                return val.encode('utf-8')
            return val

        return ''.join(
            ['[',
             ', '.join(['%s: %s' % (attr.upper(), getattr_unicode(attr))
                        for attr in self.__slots__
                        if hasattr(self, attr)]),
             ']']
        )

    def _to_json(self, json_object):
        if isinstance(json_object, SlotObject):
            d = {}
            for attr in self.__slots__:
                if hasattr(self, attr) and isinstance(getattr(self, attr), basestring):
                    value = getattr(self, attr)
                    d[attr] = value
            return d

        return str(json_object)

    def to_json(self):
        dumps = json.dumps(self, default=self._to_json)
        return dumps

    def db_values(self, mapping=None):
        """
        Словарь значений для вставки в базу данных.
        :param mapping: Словарь трансформации аттрибутов с другие имена
        :return:
        """
        values = {}
        if mapping is None:
            mapping = {}
        for attr in self.__slots__:
            if hasattr(self, attr):
                val = getattr(self, attr)
                if hasattr(val, 'db_value'):
                    val = val.db_value()
                    if val:
                        if attr in mapping:
                            values[mapping[attr]] = val
                        else:
                            values[attr] = val
                elif val:
                    if attr in mapping:
                        values[mapping[attr]] = val
                    else:
                        values[attr] = val
        return values

    __getitem__ = object.__getattribute__
    __setitem__ = object.__setattr__


class Token(SlotObject):
    # application — внутренний id приложения
    __slots__ = ('application', 'value', 'secret', 'scope', 'expires', 'token_id')


class Birthday(SlotObject):
    __slots__ = ('year', 'day', 'month')

    def __init__(self, *args, **kwargs):
        super(Birthday, self).__init__(**kwargs)

        if args and len(args) == 1 and isinstance(args[0], basestring):
            result = re.search(r'(?P<year>\d{4})-(?P<month>\d{2})-(?P<day>\d{2})', args[0])

            if not result:
                raise AttributeError('Invalid value for birthday')

            self.year = int(result.group('year')) or None
            self.month = int(result.group('month')) or None
            self.day = int(result.group('day')) or None

    def __str__(self):
        month = str(self.month).rjust(2, '0')
        day = str(self.day).rjust(2, '0')
        if self.year:
            return "%s-%s-%s" % (self.year, month, day)

        return "0000-%s-%s" % (month, day)

    def db_value(self):
        return self.__str__()


class Person(SlotObject):
    """Личные данные пользователя, полученные из социального профиля"""
    __slots__ = ('firstname', 'lastname', 'username', 'nickname', 'gender', 'birthday',
                 'email', 'phone', 'country', 'city')


class ProfileEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, SlotObject):
            d = {}
            for attr in obj.__slots__:
                if hasattr(obj, attr):
                    value = getattr(obj, attr)

                    if hasattr(value, '_to_json'):
                        d[attr] = value._to_json(value)
                    elif isinstance(value, list):
                        # tokens not produced
                        for el in value:
                            if hasattr(el, '_to_json'):
                                d[attr] = el._to_json(el)
                            else:
                                d[attr] = str(el)
                    elif value is None:
                        continue
                    else:
                        d[attr] = value
            return d
        return super(ProfileEncoder, self).default(obj)


class Profile(SlotObject):
    """Профайл пользователя в одной из социальных сетей"""
    __slots__ = ('provider', 'uid', 'userid', 'username', 'person', 'token')

    def __init__(self, **kwargs):
        super(Profile, self).__init__(**kwargs)

    def to_json(self):
        dumps = json.dumps(self, cls=ProfileEncoder)
        return dumps
