# -*- coding: utf-8 -*-

from functools import partial
import logging
import sys
from traceback import (
    format_exception_only,
    format_stack,
    format_tb,
)

import gevent
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.misc import (
    in_gevent,
    trim_message,
)


logger = logging.getLogger('subprocess.error')


def execute_multiple_methods(functions):
    retvals = []

    if in_gevent():
        # Параллельно исполняет список данных функций.
        # Журналирует отказ каждой функции в специальный журнал.
        # Наружу выбрасывается отказ первой функции из списка.

        jobs = []
        for func in functions:
            func = partial(_exc_to_retval, func)
            # gevent создаёт сопроцесс с чистым стешом, чтобы реализовать
            # наследования стеша нужно его заново наполнить данными отцовского
            # процесса.
            func = partial(_call_with_stash, func, request_ctx.save())

            job = gevent.spawn(func)
            jobs.append(job)

        gevent.joinall(jobs)

        exceptions = [j.value['exception'] for j in jobs if j.value['exception'] is not None]
        _log_greenlet_exceptions(exceptions)
        if exceptions:
            _, excval, _ = exceptions[0]
            raise excval
        else:
            retvals[:] = [j.value['value'] for j in jobs]
    else:
        for func in functions:
            retvals.append(func())

    return retvals


def _call_with_stash(func, stash_dict):
    request_ctx.load(stash_dict)
    try:
        return func()
    finally:
        # Чистим память гринлета, чтобы не было утечек
        request_ctx.clear()


def _exc_to_retval(func):
    try:
        retval = func()
        retexc = None
    except:
        retval = None
        retexc = sys.exc_info()
    return {'exception': retexc, 'value': retval}


def _log_greenlet_exceptions(exceptions):
    for exc in exceptions:
        if exc is not None:
            logger.debug(trim_message(_format_greenlet_exception(exc), cut=False))


def _format_greenlet_exception(exc):
    exctype, excval, exctb = exc
    lines = [
        'Greenlet failed: %s\n' % exctype.__name__,
        'Traceback (most recent call last):\n',
    ]
    lines += format_stack()
    lines += format_tb(exctb)
    lines += format_exception_only(exctype, excval)
    return ''.join(lines)
